import { z } from 'zod';

const objectIdRegex = /^[a-fA-F0-9]{24}$/;
const uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

/**
 * Validates MongoDB ObjectId (24 hex chars)
 */
export const objectIdSchema = z
    .string()
    .trim()
    .refine((val) => objectIdRegex.test(val), {
        message: 'The ID is in an invalid format',
    });

/**
 * Validates UUID (versions 1-5)
 */
export const uuidSchema = z
    .string()
    .trim()
    .refine((val) => uuidRegex.test(val), {
        message: 'The ID is in an invalid format',
    });

/**
 * Generic ID schema accepting either a Mongo ObjectId or a UUID.
 */
export const idSchema = z.union([objectIdSchema, uuidSchema]);

export const idArraySchema = z.array(idSchema).nonempty('Provide at least one valid ID');

export const objectIdArraySchema = z.array(objectIdSchema).nonempty('Provide at least one valid ID');
