import z from "zod";

/**
 * Collection of Zod schemas for validating strings with different character sets.
 */
export const stringSchemas = {
    /**
     * Validates strings containing only letters (uppercase and lowercase) and spaces.
     * Useful for fields like simple "first_name", "last_name".
     */
    lettersAndSpaces: z
        .string()
        .trim()
        .min(1, 'Field cannot be empty.')
        .regex(/^[a-zA-Z\s]+$/, 'Field can only contain letters and spaces.'),

    /**
     * Validates strings containing letters (uppercase and lowercase), numbers, and spaces.
     * Useful for fields like simplified addresses, codes.
     */
    alphanumericAndSpaces: z
        .string()
        .trim()
        .min(1, 'Field cannot be empty.')
        .regex(/^[a-zA-Z0-9\s]+$/, 'Field can only contain letters, numbers, and spaces.'),

    /**
     * Validates strings containing letters (including accented characters), and spaces.
     * Ideal for full names in Portuguese.
     */
    portugueseName: z
        .string()
        .trim()
        .min(1, 'Name cannot be empty.')
        .max(200, 'Name cannot exceed 200 characters.')
        .regex(/^[a-zA-ZÀ-ÿ\s']+$/, 'Name can only contain letters, accented characters, and spaces.'),

    /**
     * Validates strings containing letters (including accented characters), numbers, spaces,
     * and some common special characters (hyphen, period, comma, apostrophe).
     * Suitable for descriptions, titles, or more general text fields.
     */
    extendedText: z
        .string()
        .trim()
        .min(1, 'Field cannot be empty.')
        .max(500, 'Field cannot exceed 500 characters.')
        .regex(/^[a-zA-ZÀ-ÿ0-9\s\-,.'’]+$/, 'Field contains invalid characters.'),

    /**
     * Validates strings that are valid URLs.
     * Useful for fields like image_profile, links, etc.
     */
    url: z
        .string()
        .trim()
        .url('Provide a valid URL.'),
};