import { validate as validateUUID, version as uuidVersion } from 'uuid';
import { z } from 'zod';

const uuidV7Regex =
    /^[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-7[0-9a-fA-F]{3}-[89abAB][0-9a-fA-F]{3}-[0-9a-fA-F]{12}$/;

/**
 * Validate if string is uuidv7
 */
function isUUIDv7(value: string): boolean {
    return (
        uuidV7Regex.test(value) &&
        validateUUID(value) &&
        uuidVersion(value) === 7
    );
}

/**
 * Schema Zod que valida UUIDv7 individuais.
 */
export const uuidV7Schema = z
    .string()
    .trim()
    .refine(isUUIDv7, {
        message: 'The ID is in an invalid format',
    });

/**
 * Schema Zod que valida arrays de UUIDv7.
 */
export const uuidV7ArraySchema = z
    .array(uuidV7Schema)
    .nonempty('Provide at least one valid ID');