import { z } from 'zod';
import { uuidV7Schema } from './shared/uuidv7.schema';
import { stringSchemas } from './shared/simpleString.schema';
import { dateValidation } from './shared/date.schema';
import { passwordValidation } from './shared/password.schema';
import { emailValidation } from './shared/email.schema';
import { cpfValidation } from './shared/cpf.schema';
import { rgValidation } from './shared/rg.schema';
import { telephoneValidation } from './shared/telephone.schema';
import { classSchemas } from './class.schema';

/**
 * Utility function to validate credit card number
 */
function isValidCreditCard(cardNumber: string): boolean {
    const cleaned = cardNumber.replace(/\s/g, '');
    return /^\d{13,19}$/.test(cleaned);
}

/**
 * Utility function to validate CVV
 */
function isValidCVV(cvv: string): boolean {
    return /^\d{3,4}$/.test(cvv);
}

/**
 * Utility function to validate expiration date (MM/YY)
 */
function isValidExpiration(expiration: string): boolean {
    if (!/^\d{2}\/\d{2}$/.test(expiration)) return false;

    const [month, year] = expiration.split('/').map(Number);
    if (month < 1 || month > 12) return false;

    const currentYear = new Date().getFullYear() % 100;
    const currentMonth = new Date().getMonth() + 1;

    if (year < currentYear) return false;
    if (year === currentYear && month < currentMonth) return false;

    return true;
}

/**
 * Shared fields for student-related payloads
 */
export const studentSchema = {
    full_name: stringSchemas.portugueseName,

    birth_date: dateValidation.isoDateSchema
        .transform(val => new Date(val))
        .pipe(
            z.date()
                .max(new Date(new Date().setFullYear(new Date().getFullYear() - 3)), 'Student must be at least 3 years old.')
                .min(new Date(new Date().setFullYear(new Date().getFullYear() - 100)), 'Student cannot be older than 100 years.')
        ),

    gender: z
        .enum(['male', 'female', 'other'])
        .optional(),

    email: emailValidation,

    password: passwordValidation,

    cpf: z.object({
        value: cpfValidation,
        upload_id: uuidV7Schema.optional(),
    }),

    rg: z.object({
        value: rgValidation,
        upload_id: uuidV7Schema.optional(),
    }).optional(),

    telephone: telephoneValidation,

    academic: z.object({
        current_grade: z
            .string()
            .trim()
            .min(1, 'Current grade cannot be empty.')
            .max(50, 'Current grade cannot exceed 50 characters.'),
        school: z
            .string()
            .trim()
            .min(1, 'School cannot be empty if provided.')
            .max(100, 'School cannot exceed 100 characters.')
            .optional(),
        subjects: z
            .array(z.string().trim().min(1, 'Subject cannot be empty.'))
            .min(1, 'Provide at least one subject.')
            .max(10, 'Cannot exceed 10 subjects.'),
        difficulty_level: z
            .enum(['beginner', 'intermediate', 'advanced']),
        main_goal: z
            .string()
            .trim()
            .min(1, 'Main goal cannot be empty.')
            .max(200, 'Main goal cannot exceed 200 characters.'),
    }),

    class_preferences: z.object({
        modality: z.enum(['online', 'presencial', 'hibrida']),
        frequency: z
            .string()
            .trim()
            .min(1, 'Frequency cannot be empty.')
            .max(50, 'Frequency cannot exceed 50 characters.'),
        available_times: z
            .array(z.object({
                day_of_week: z
                    .number()
                    .int()
                    .min(0, 'Day of week must be between 0 and 6.')
                    .max(6, 'Day of week must be between 0 and 6.'),
                start_time: z
                    .number()
                    .int()
                    .min(0, 'Start time must be between 0 and 1439.')
                    .max(1439, 'Start time must be between 0 and 1439.'),
                end_time: z
                    .number()
                    .int()
                    .min(0, 'End time must be between 0 and 1439.')
                    .max(1439, 'End time must be between 0 and 1439.'),
            }))
            .min(1, 'Provide at least one available time.')
            .refine(times => {
                return times.every(time => time.end_time > time.start_time);
            }, 'End time must be after start time for all available times.'),
        preferred_teacher: uuidV7Schema.optional(),
    }),

    guardian: z.object({
        full_name: z
            .string()
            .trim()
            .min(1, 'Guardian full name cannot be empty.')
            .max(200, 'Guardian full name cannot exceed 200 characters.'),
        birth_date: z
            .string()
            .or(z.date())
            .transform(val => new Date(val))
            .pipe(
                z.date()
                    .max(new Date(new Date().setFullYear(new Date().getFullYear() - 18)), 'Guardian must be at least 18 years old.')
                    .min(new Date(new Date().setFullYear(new Date().getFullYear() - 100)), 'Guardian cannot be older than 100 years.')
            ),
        telephone: telephoneValidation,
        email: emailValidation,
        relationship: z
            .string()
            .trim()
            .min(1, 'Relationship cannot be empty.')
            .max(50, 'Relationship cannot exceed 50 characters.'),
        cpf: z.object({
            value: cpfValidation,
            upload_id: uuidV7Schema.optional(),
        }),
        rg: z.object({
            value: rgValidation,
            upload_id: uuidV7Schema.optional(),
        }),
    }).optional(),

    financial: z.object({
        credit_card_registered: z.boolean(),
        card_info: z.object({
            cardholder_name: z
                .string()
                .trim()
                .min(1, 'Cardholder name cannot be empty.')
                .max(100, 'Cardholder name cannot exceed 100 characters.')
                .regex(/^[a-zA-ZÀ-ÿ\s']+$/, 'Cardholder name can only contain letters and spaces.'),
            card_number: z
                .string()
                .trim()
                .refine(isValidCreditCard, 'Provide a valid credit card number.'),
            expiration: z
                .string()
                .trim()
                .refine(isValidExpiration, 'Provide a valid expiration date (MM/YY).'),
            cvv: z
                .string()
                .trim()
                .refine(isValidCVV, 'Provide a valid CVV (3 or 4 digits).'),
        }).optional(),
    }).optional(),
};

/**
 * Schema for creating a new student
 */
const createStudentBodySchema = z.object({
    full_name: studentSchema.full_name,
    birth_date: studentSchema.birth_date,
    gender: studentSchema.gender,
    email: studentSchema.email,
    password: studentSchema.password,
    cpf: studentSchema.cpf,
    rg: studentSchema.rg,
    telephone: studentSchema.telephone,
    academic: studentSchema.academic,
    class_preferences: studentSchema.class_preferences,
    guardian: studentSchema.guardian,
    financial: studentSchema.financial,
});

/**
 * Schema for updating an existing student
 */
const updateStudentBodySchema = z
    .object({
        birth_date: studentSchema.birth_date.optional(),
        gender: studentSchema.gender,
        cpf: studentSchema.cpf.optional(),
        rg: studentSchema.rg,
        telephone: studentSchema.telephone.optional(),
        academic: studentSchema.academic.optional(),
        class_preferences: studentSchema.class_preferences.optional(),
        guardian: studentSchema.guardian.optional(),
        financial: studentSchema.financial.optional(),
    })
    .superRefine((data, ctx) => {
        if (Object.keys(data).length === 0) {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'Provide at least one field to update.',
                path: [],
            });
        }

        // Validate that if academic is provided, all required fields are present
        if (data.academic && Object.keys(data.academic).length > 0) {
            if (!data.academic.current_grade || !data.academic.subjects || !data.academic.difficulty_level || !data.academic.main_goal) {
                ctx.addIssue({
                    code: z.ZodIssueCode.custom,
                    message: 'When updating academic info, provide all required fields: current_grade, subjects, difficulty_level, main_goal.',
                    path: ['academic'],
                });
            }
        }

        // Validate that if class_preferences is provided, all required fields are present
        if (data.class_preferences && Object.keys(data.class_preferences).length > 0) {
            if (!data.class_preferences.modality || !data.class_preferences.frequency || !data.class_preferences.available_times) {
                ctx.addIssue({
                    code: z.ZodIssueCode.custom,
                    message: 'When updating class preferences, provide all required fields: modality, frequency, available_times.',
                    path: ['class_preferences'],
                });
            }
        }
    });

/**
 * Schema for query parameters when searching for teachers
 */
const getTeachersQuerySchema = z.object({
    days_of_week: z
        .any()
        .transform(val => {
            let result = [];

            if (val === null || val === undefined) {
                return val;
            }

            if (Array.isArray(val)) {
                result = val.map(item => String(item).replace(/['"]/g, ''));
            } else if (typeof val === 'string') {
                try {
                    const parsed = JSON.parse(val);
                    if (Array.isArray(parsed)) {
                        result = parsed.map(item => String(item).replace(/['"]/g, ''));
                    } else {
                        result = [String(parsed).replace(/['"]/g, '')];
                    }
                } catch {
                    if (val.includes(',')) {
                        result = val.split(',').map(day => day.trim().replace(/['"]/g, ''));
                    } else {
                        result = [val.replace(/['"]/g, '')];
                    }
                }
            } else {
                result = [String(val).replace(/['"]/g, '')];
            }

            return result;
        })
        .pipe(
            z.array(z.string())
                .min(1, 'Provide at least one day of week.')
                .refine(days => {
                    const validDays = ['0', '1', '2', '3', '4', '5', '6'];
                    return days.every(day => validDays.includes(day));
                }, 'Days of week must be numbers between 0 and 6.')
        )
        .optional(),
    start_time: z
        .string()
        .transform(val => parseInt(val, 10))
        .pipe(
            z.number()
                .int()
                .min(0, 'Start time must be between 0 and 1439.')
                .max(1439, 'Start time must be between 0 and 1439.')
        )
        .optional(),
    end_time: z
        .string()
        .transform(val => parseInt(val, 10))
        .pipe(
            z.number()
                .int()
                .min(0, 'End time must be between 0 and 1439.')
                .max(1439, 'End time must be between 0 and 1439.')
        )
        .optional(),
    discipline_id: uuidV7Schema.optional(),
})
    .refine(data => {
        if (data.start_time && data.end_time) {
            return data.end_time > data.start_time;
        }
        return true;
    }, 'End time must be after start time.');

const listAdaptedStudentsQuery = z.object({
    daysOfWeek: z
        .any()
        .transform(val => {
            let result = [];

            if (val === null || val === undefined) {
                return val;
            }

            if (Array.isArray(val)) {
                result = val.map(item => String(item).replace(/['"]/g, ''));
            } else if (typeof val === 'string') {
                try {
                    const parsed = JSON.parse(val);
                    if (Array.isArray(parsed)) {
                        result = parsed.map(item => String(item).replace(/['"]/g, ''));
                    } else {
                        result = [String(parsed).replace(/['"]/g, '')];
                    }
                } catch {
                    if (val.includes(',')) {
                        result = val.split(',').map(day => day.trim().replace(/['"]/g, ''));
                    } else {
                        result = [val.replace(/['"]/g, '')];
                    }
                }
            } else {
                result = [String(val).replace(/['"]/g, '')];
            }

            return result;
        })
        .pipe(
            z.array(z.string())
                .min(1, 'Provide at least one day of week.')
                .refine(days => {
                    const validDays = ['0', '1', '2', '3', '4', '5', '6'];
                    return days.every(day => validDays.includes(day));
                }, 'Days of week must be numbers between 0 and 6.')
        )
        .optional(),
    startMinutes: z
        .string()
        .transform(val => parseInt(val, 10))
        .pipe(
            z.number()
                .int()
                .min(0, 'Start time must be between 0 and 1439.')
                .max(1439, 'Start time must be between 0 and 1439.')
        )
        .optional(),
    endMinutes: z
        .string()
        .transform(val => parseInt(val, 10))
        .pipe(
            z.number()
                .int()
                .min(0, 'End time must be between 0 and 1439.')
                .max(1439, 'End time must be between 0 and 1439.')
        )
        .optional(),
})
    .refine(data => {
        if (data.startMinutes && data.endMinutes) {
            return data.endMinutes > data.startMinutes;
        }
        return true;
    }, 'End time must be after start time.');

/**
 * Collection of schemas responsible for validating all student module requests.
 */
export const studentSchemas = {
    createStudent: z.object({
        body: createStudentBodySchema,
    }),
    getStudentById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    listAdaptedStudents: z.object({
        query: listAdaptedStudentsQuery,
    }),
    updateStudent: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
        body: updateStudentBodySchema,
    }),
    deleteStudent: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    getTeachers: z.object({
        query: getTeachersQuerySchema,
    }),
    createClass: classSchemas.createClass,
    getClasses: classSchemas.getClasses,
    getClassById: classSchemas.getClassById,
    updateClass: classSchemas.updateClass,
};

/**
 * Type for student schemas dependency injection
 */
export type IStudentSchemas = typeof studentSchemas;