import { z } from 'zod';
import { uuidV7ArraySchema, uuidV7Schema } from './shared/uuidv7.schema';
import { emailValidation } from './shared/email.schema';
import { passwordValidation } from './shared/password.schema';
import { stringSchemas } from './shared/simpleString.schema';
import { cpfValidation } from './shared/cpf.schema';
import { rgValidation } from './shared/rg.schema';
import { telephoneValidation } from './shared/telephone.schema';
import { dateValidation } from './shared/date.schema';
import { zipCodeValidation } from './shared/cep.schema';
import { getClassesQuerySchema } from './class.schema';
import { cnpjValidation } from './shared/cnpj.schema';

/**
 * Utility function to validate Brazilian state (UF)
 */
function isValidState(state: string): boolean {
    const validStates = [
        'AC', 'AL', 'AP', 'AM', 'BA', 'CE', 'DF', 'ES', 'GO', 'MA',
        'MT', 'MS', 'MG', 'PA', 'PB', 'PR', 'PE', 'PI', 'RJ', 'RN',
        'RS', 'RO', 'RR', 'SC', 'SP', 'SE', 'TO'
    ];
    return validStates.includes(state.toUpperCase());
}

/**
 * Utility function to validate payment type and key (PIX, bank account, etc.)
 */
function isValidPaymentInfo(type: string, key: string): boolean {
    const validTypes = ['pix', 'bank_account', 'digital_wallet'];

    if (!validTypes.includes(type)) {
        return false;
    }

    // Basic validation for PIX key (can be CPF, email, phone, or random key)
    if (type === 'pix') {
        // PIX keys can be: CPF/CNPJ, email, phone, or random key
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        const phoneRegex = /^\+55\d{10,11}$/;
        const cpfCnpjRegex = /^\d{11,14}$/;

        return emailRegex.test(key) || phoneRegex.test(key) || cpfCnpjRegex.test(key) || key.length >= 10;
    }

    // For bank account, validate it's not empty
    if (type === 'bank_account') {
        return key.length >= 5;
    }

    // For digital wallet, validate it's not empty
    if (type === 'digital_wallet') {
        return key.length >= 5;
    }

    return true;
}

/**
 * Shared fields for teacher-related payloads
 */
export const teacherSchema = {
    full_name: stringSchemas.portugueseName,

    cpfCnpj: z.object({
        value: z
            .union([cpfValidation, cnpjValidation])
            .refine(() => true, { message: "Invalid CPF/CNPJ format" }),
        upload_id: uuidV7Schema.optional(),
        type: z.enum(['CPF', 'CNPJ']),
    }),

    rg: z.object({
        value: rgValidation,
        upload_id: uuidV7Schema.optional(),
    }).optional(),

    email: emailValidation,

    password: passwordValidation,

    telephone: telephoneValidation,

    birth_date: dateValidation.isoDateSchema
        .transform((val) => new Date(val)) // Transform to Date object after validation
        .pipe(
            z.date()
                .max(new Date(new Date().setFullYear(new Date().getFullYear() - 18)), 'Teacher must be at least 18 years old.')
                .min(new Date(new Date().setFullYear(new Date().getFullYear() - 100)), 'Teacher cannot be older than 100 years.')
        ),

    image_profile: z
        .string()
        .trim()
        .url('Provide a valid URL for profile image.')
        .optional(),

    address: z.object({
        street: z
            .string()
            .trim()
            .min(1, 'Street cannot be empty.')
            .max(200, 'Street cannot exceed 200 characters.'),
        number: z
            .string()
            .trim()
            .min(1, 'Number cannot be empty.')
            .max(10, 'Number cannot exceed 10 characters.'),
        complement: z
            .string()
            .trim()
            .max(100, 'Complement cannot exceed 100 characters.')
            .optional(),
        city: z
            .string()
            .trim()
            .min(1, 'City cannot be empty.')
            .max(100, 'City cannot exceed 100 characters.'),
        state: z
            .string()
            .trim()
            .length(2, 'State must be 2 characters (UF).')
            .refine(isValidState, 'Provide a valid Brazilian state (UF).'),
        zip_code: zipCodeValidation,
    }).optional(),

    education: z.object({
        level: z
            .string()
            .trim()
            .min(1, 'Education level cannot be empty.')
            .max(100, 'Education level cannot exceed 100 characters.'),
        field: z
            .string()
            .trim()
            .min(1, 'Field of study cannot be empty.')
            .max(100, 'Field of study cannot exceed 100 characters.'),
        teaching_experience: z
            .string()
            .trim()
            .min(1, 'Teaching experience cannot be empty.')
            .max(50, 'Teaching experience cannot exceed 50 characters.'),
        disciplines: uuidV7ArraySchema,
        education_levels: z
            .array(
                z.string()
                    .trim()
                    .min(1, 'Education level cannot be empty.')
                    .max(50, 'Education level cannot exceed 50 characters.')
            )
            .min(1, 'Provide at least one education level.')
            .max(10, 'Cannot exceed 10 education levels.'),
    }).optional(),

    availability: z.object({
        in_person_location: z
            .array(
                z.string()
                    .trim()
                    .min(1, 'Location cannot be empty.')
                    .max(100, 'Location cannot exceed 100 characters.')
            )
            .min(1, 'Provide at least one in-person location.')
            .max(5, 'Cannot exceed 5 locations.'),
        available_times: z
            .array(z.object({
                day_of_week: z
                    .number()
                    .int()
                    .min(0, 'Day of week must be between 0 and 6.')
                    .max(6, 'Day of week must be between 0 and 6.'),
                start_time: z
                    .number()
                    .int()
                    .min(0, 'Start time must be between 0 and 1439.')
                    .max(1439, 'Start time must be between 0 and 1439.'),
                end_time: z
                    .number()
                    .int()
                    .min(0, 'End time must be between 0 and 1439.')
                    .max(1439, 'End time must be between 0 and 1439.'),
            }))
            .min(1, 'Provide at least one available time.')
            .refine(times => {
                return times.every(time => time.end_time > time.start_time);
            }, 'End time must be after start time for all available times.'),
        accepts_emergency_classes: z.boolean().default(false),
    }).optional(),

    payment: z.object({
        hourly_rate: z
            .number()
            .positive('Hourly rate must be positive.')
            .min(10, 'Hourly rate must be at least R$ 10.00.')
            .max(1000, 'Hourly rate cannot exceed R$ 1000.00.'),
        payment_info: z.object({
            type: z.enum(['pix', 'bank_account', 'digital_wallet']),
            key: z
                .string()
                .trim()
                .min(1, 'Payment key cannot be empty.')
                .max(200, 'Payment key cannot exceed 200 characters.'),
        }).refine(
            (data) => isValidPaymentInfo(data.type, data.key),
            {
                message: 'Provide valid payment information for the selected type.',
                path: ['key'],
            }
        ),
    }).optional(),

    preferences: z.object({
        student_type_preference: z
            .string()
            .trim()
            .max(200, 'Student type preference cannot exceed 200 characters.')
            .optional(),
        general_observations: z
            .string()
            .trim()
            .max(500, 'General observations cannot exceed 500 characters.')
            .optional(),
    }).optional(),
};

/**
 * Schema for creating a new teacher
 */
const createTeacherBodySchema = z.object({
    full_name: teacherSchema.full_name,
    cpfCnpj: teacherSchema.cpfCnpj,
    rg: teacherSchema.rg,
    email: teacherSchema.email,
    password: teacherSchema.password,
    telephone: teacherSchema.telephone,
    birth_date: teacherSchema.birth_date,
    image_profile: teacherSchema.image_profile,
    address: teacherSchema.address,
    education: teacherSchema.education,
    availability: teacherSchema.availability,
    payment: teacherSchema.payment,
    preferences: teacherSchema.preferences,
});

/**
 * Schema for updating an existing teacher
 */
const updateTeacherBodySchema = z
    .object({
        full_name: teacherSchema.full_name.optional(),
        cpfCnpj: teacherSchema.cpfCnpj.optional(),
        rg: teacherSchema.rg.optional(),
        telephone: teacherSchema.telephone.optional(),
        birth_date: teacherSchema.birth_date.optional(),
        image_profile: teacherSchema.image_profile.optional(),
        address: teacherSchema.address.optional(),
        education: teacherSchema.education.optional(),
        availability: teacherSchema.availability.optional(),
        payment: teacherSchema.payment.optional(),
        preferences: teacherSchema.preferences.optional(),
    })
    .superRefine((data, ctx) => {
        if (Object.keys(data).length === 0) {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'Provide at least one field to update.',
                path: [],
            });
        }

        // Validate that if education is provided, all required fields are present
        if (data.education && Object.keys(data.education).length > 0) {
            const requiredFields = ['level', 'field', 'teaching_experience', 'disciplines', 'education_levels'];
            const missingFields = requiredFields.filter(field => !data.education?.[field as keyof typeof data.education]);

            if (missingFields.length > 0) {
                ctx.addIssue({
                    code: z.ZodIssueCode.custom,
                    message: `When updating education, provide all required fields: ${requiredFields.join(', ')}.`,
                    path: ['education'],
                });
            }
        }

        // Validate that if availability is provided, all required fields are present
        if (data.availability && Object.keys(data.availability).length > 0) {
            const requiredFields = ['available_times'];
            const missingFields = requiredFields.filter(field => !data.availability?.[field as keyof typeof data.availability]);

            if (missingFields.length > 0) {
                ctx.addIssue({
                    code: z.ZodIssueCode.custom,
                    message: `When updating availability, provide all required fields: ${requiredFields.join(', ')}.`,
                    path: ['availability'],
                });
            }
        }

        // Validate that if payment is provided, all required fields are present
        if (data.payment && Object.keys(data.payment).length > 0) {
            if (!data.payment.hourly_rate || !data.payment.payment_info) {
                ctx.addIssue({
                    code: z.ZodIssueCode.custom,
                    message: 'When updating payment, provide both hourly_rate and payment_info.',
                    path: ['payment'],
                });
            }
        }
    });

/**
 * Query parameters for filtering classes to teacher
 */
const listClassesQuerySchema = getClassesQuerySchema
    .omit({
        student_id: true,
    })
    .required({
        teacher_id: true,
    });

/**
 * Collection of schemas responsible for validating all teacher module requests.
 */
export const teacherSchemas = {
    createTeacher: z.object({ 
        body: createTeacherBodySchema,
    }),
    getTeacherById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    updateTeacher: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
        body: updateTeacherBodySchema,
    }),
    deleteTeacher: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    getClasses: z.object({
        query: listClassesQuerySchema,
    }),
    getClassById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
};

/**
 * Type for teacher schemas dependency injection
 */
export type ITeacherSchemas = typeof teacherSchemas;