import { z } from 'zod';
import {
    uuidV7Schema,
    uuidV7ArraySchema
} from './shared/uuidv7.schema';
import { emailValidation } from './shared/email.schema';
import { passwordValidation } from './shared/password.schema';
import { stringSchemas } from './shared/simpleString.schema';
import { studentSchemas } from './student.schema';
import { teacherSchemas } from './teacher.schema';

/**
 * Schema used to validate user registration payloads (matches IParamsCreateUser).
 */
export const userRegisterSchema = z.object({
    name: stringSchemas.portugueseName.optional(),
    email: emailValidation,
    password: passwordValidation,
});

/**
 * Schema that validates a non-empty list of role IDs
 */
const rolesIdsSchema = uuidV7ArraySchema.min(1, {
    message: 'Provide at least one role ID',
});

/**
 * Schema describing the payload accepted when updating an existing user.
 */
const updateUserBodySchema = z
    .object({
        name: stringSchemas.portugueseName.optional(),
        is_active: z.boolean().optional(),
        roles: z.array(uuidV7Schema).optional(),
    })
    .superRefine((data, ctx) => {
        if (Object.keys(data).length === 0) {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'Provide at least one field to update.',
                path: [],
            });
        }
    });

/**
 * Collection of schemas responsible for validating all user module requests.
 */
export const userSchemas = {
    createUser: z.object({
        body: userRegisterSchema.extend({
            name: z
                .string()
                .trim()
                .min(1, 'Name cannot be empty.'),
        }),
    }),
    listUsers: z.object({
        query: z
            .object({
                name: z.string().trim().optional(),
                email: z.string().trim().optional(),
                is_active: z
                    .enum(['true', 'false'])
                    .transform(value => value === 'true')
                    .optional(),
            })
            .default({}),
    }),
    getUserById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    getUserRolesById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    assignRoleToUser: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
        body: z.object({
            rolesIds: rolesIdsSchema,
        }),
    }),
    updateUser: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
        body: updateUserBodySchema,
    }),
    removeRoleFromUser: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
        body: z.object({
            rolesIds: rolesIdsSchema,
        }),
    }),
    deleteUser: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    ...studentSchemas,
    ...teacherSchemas,
};

/**
 * Type for user schemas dependency injection
 */
export type IUserSchemas = typeof userSchemas;