// eslint-disable-next-line @typescript-eslint/triple-slash-reference
/// <reference path="../types/index.d.ts"/>
import { IRouter } from '../shared/contracts/IRouter';
import { DatabaseService } from '../infrastructure/database/db.service';
import { SwaggerService } from '../shared/services/swagger.service';
import { AppError } from '../shared/utils/customError';
import { ContextAsyncHooks, Logger } from 'traceability';
import { Server as httpServer } from 'http';
import rateLimit from 'express-rate-limit';
import helmet from 'helmet';
import cors from 'cors';
import express, {
    Request,
    Response,
    Application,
    NextFunction,
    RequestHandler,
} from 'express';
import "express-async-errors";
import { ZodError } from 'zod';
import * as winston from 'winston';

export class Server {
    // Application data
    private readonly app: Application;
    private readonly port: number;
    private readonly basePath: string;
    private readonly routes: IRouter[];
    private readonly timeoutMilliseconds: number;

    // Auxiliary services
    private readonly databaseService: DatabaseService;
    private readonly swaggerService: SwaggerService;

    // Application protection middlewares
    private readonly protectMiddlewares = [
        ContextAsyncHooks.getExpressMiddlewareTracking(),
        rateLimit({
            windowMs: 1000, // 1s
            max: 5, // rate limit 5 requests
            message: {
                message: 'Request rate limit (burst). Please try again later.'
            },
            standardHeaders: false,
            legacyHeaders: false,
        }),
        rateLimit({
            windowMs: 60 * 1000, // 1m
            max: 50, // rate limit 50 requests
            message: {
                message: 'Request rate limit (sustained). Please try again later.'
            },
            standardHeaders: true,
            legacyHeaders: false,
        }),
        helmet(),
        cors(),
        express.json({ limit: '3mb' }),
        express.urlencoded({ limit: '3mb', extended: true }),
    ];

    constructor (
        appInit: {
            basePath: string;
            port: number;
            routers?: Array<IRouter>;
            timeoutMilliseconds: number;
        }
    ) {
        globalThis.AppError = AppError;

        // Configure Logger to json format
        const consoleTransport = Logger.transports.find(
            (transport) => transport instanceof winston.transports.Console
        );
        if (consoleTransport) {
            consoleTransport.format = winston.format.combine(
                winston.format.timestamp(),
                winston.format.json({ space: 2 })
            );
        }

        this.app = express();
        this.port = appInit.port;
        this.basePath = appInit.basePath;
        this.routes = appInit.routers || [];
        this.timeoutMilliseconds = appInit.timeoutMilliseconds;
        this.databaseService = new DatabaseService();
        this.swaggerService = new SwaggerService(this.app, this.basePath);
    }

    /**
     * Start the server
     */
    public async start() {
        try {
            // Database connection
            await this.databaseService.connect();

            // Middlewares
            this.middlewares(this.protectMiddlewares);

            // Request logger middlewares
            this.requestLogger();

            // Swagger documentation
            this.swaggerService.setupSwaggerWebDocs();

            // Swagger validator router
            this.swaggerService.swaggerValidator();

            // Test server health
            this.app.get(
                `${this.basePath}/health`,
                (
                    req: Request,
                    res: Response
                ) => {
                    res.status(200).json({ status: 'OK' });
                }
            );

            // Configure routers
            this.configureRouters(this.routes);

            // Handler not found paths
            this.notFoundHandler();
            
            // Swagger error manipulator
            this.app.use(this.swaggerService.getErrorHandler());

            // Generic error manipulator
            this.errorHandler();

            // Init application
            this.listen();
        } catch(error) {
            Logger.error(`Failed to start application: ${(error as Error).message}`);
        }
    }

    /**
     * Configure the application middlewares
     */
    private middlewares (
        middleWares: Array<RequestHandler>
    ) {
        middleWares.forEach(
            (middleWare) => this.app.use(middleWare)
        );
    }

    /**
     * Error handler
     */
    private errorHandler() {
        this.app.use((
                err: Error | AppError,
                req: Request,
                res: Response,
                _next: NextFunction
            ) => {
                // Correctly handles and returns the error
                if (err instanceof ZodError) {

                    // Validation error with Zod
                    res.status(400).json({
                        message: 'Validation failed',
                        issues: err.issues.map(issue => ({
                            path: issue.path.join('.'),
                            message: issue.message,
                        })),
                    });

                } else if (err instanceof AppError && err.status !== 500) {

                    // Application error
                    res.status(err.status).json({ message: err.message });

                } else {

                    // Internal server error
                    // Only logs generic server errors
                    console.log(err);
                    res.status(500).json({ message: `Internal server error` });
                }
            }
        );
    }

    /**
     * Request logger
     */
    private requestLogger() {
        this.app.use((
                req: Request,
                res: Response,
                next: NextFunction,
            ) => {
                // Saves the message sent to the user
                const originalJson = res.json;
                res.json = (body) => {
                    if (body && body.message) {
                        res.locals.responseMessage = body.message;
                    }
                    return originalJson.apply(res, [body]);
                }

                // Logs the request data after sending the response
                res.on('finish', () => {
                    const message = res.locals.responseMessage || "";
                    Logger.info(message, {
                        method: req.method,
                        path: req.originalUrl,
                        status: res.statusCode,
                        user: req.user || "anonymous",
                        ip: req.ip,
                    });
                });

                next();
            }
        );
    }

    /**
     * Handler for undefined routes
     */
    private notFoundHandler() {
        this.app.use((
                // eslint-disable-next-line @typescript-eslint/no-unused-vars
                req: Request, res: Response, next: NextFunction,
            ) => {
                throw new AppError('Route not found', 404);
            }
        );
    }

    /**
     * Configure all the routers 
     */
    private configureRouters (
        routers: Array<IRouter>
    ) {
        routers.forEach((
                router
            ) => {
                this.app.use(
                    `${this.basePath}${router.path}`, // Declares the endpoint
                    router.getRoutes() // Declare the routes for endpoint
                );
            }
        );
    }

    /**
     * Inicialize the server for listen the requests
     */
    private listen(): httpServer {
        return this.app.listen(
            this.port,
            () => {
                Logger.info(`App listening on the http://localhost:${this.port}`);
            }
        ).setTimeout(this.timeoutMilliseconds);
    }
}