import { Request, Response } from "express";
import { ICategorySchemas } from "../../main/schemas/category.schema";
import { ICategoryService } from "./interfaces/category.service.interface";

/**
 * Controller responsible for handling category-related HTTP requests.
 * Provides endpoints for creating, retrieving, updating, and deleting categorys.
 */
export class CategoryController {
    private readonly service: ICategoryService;
    private readonly schemas: ICategorySchemas;

    constructor (
        service: ICategoryService,
        schemas: ICategorySchemas
    ) {
        this.service = service;
        this.schemas = schemas;
    }

    /**
     * Creates a new category
     */
    public createCategory = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.schemas.createCategory.parse(req);
        const { name, description } = body;

        const category = await this.service.createCategory(
            {
                name,
                description: description || "",
            }
        );

        res.status(201).json({ message: "Category created successfully", category });
    }

    /**
     * Finds a category by ID
     */
    public getCategoryById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getCategoryById.parse(req);
        const { id } = params;

        const category = await this.service.getCategoryById(id);

        res.status(200).json(category);
    }

    /**
     * Lists all categorys
     */
    public listCategories = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { filter } = req.body;

        const categorys = await this.service.listCategories(filter);

        res.status(200).json(categorys);
    }

    /**
     * Updates a category by ID
     */
    public updateCategory = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params, body } = this.schemas.updateCategory.parse(req);

        const category = await this.service.updateCategoryById(
            {
                id: params.id,
                data: body
            }
        );

        res.status(200).json({ message: "Category updated successfully", category });
    }

    /**
     * Deletes a category by ID
     */
    public deleteCategory = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getCategoryById.parse(req);

        const category = await this.service.deleteCategoryById(params.id);

        res.status(200).json({ message: "Category deleted successfully", category });
    }
}