import { ICategoryRepository } from "./interfaces/category.repository.interface";
import { ICategory } from "./interfaces/category.interface";
import { Category } from "./category.entity";
import {
    ICategoryService,
    IParamsCreateCategory,
    IParamsCategoryService,
    IParamsUpdateCategory
} from "./interfaces/category.service.interface";

/**
 * Service responsible for business logic related to Categorys.
 * Handles creation, retrieval, update, and deletion of categorys.
 */
export class CategoryService implements ICategoryService {
    private readonly repository: ICategoryRepository;

    constructor (
        { repository }: IParamsCategoryService
    ) {
        this.repository = repository;
    }

    /**
     * Creates a new category
     * @param category - Data for category creation
     * @returns The created category
     */
    public createCategory = async (
        category: IParamsCreateCategory
    ): Promise<ICategory> => {
        const newCategory = new Category(category);

        const exist = await this.repository.findCategoryByName(category.name);
        if (exist) throw new AppError('Category with name alreay exist', 400);

        return await this.repository.createCategory(newCategory);
    }

    /**
     * Finds a category by ID
     * @param id - Category ID
     * @returns The found category or throws error if not found
     */
    public getCategoryById = async (
        id: string
    ): Promise<ICategory> => {
        const category = await this.repository.findCategoryById(id);
        if (!category) throw new AppError('Category not found', 404);

        return category;
    }

    /**
     * Lists categorys according to filter
     * @param filter - Search filter
     * @returns Array of found categorys
     */
    public listCategories = async (
        filter: Partial<ICategory> = {}
    ): Promise<ICategory[]> => {

        /*
         * Correctly configure the filter
        */

        return await this.repository.listCategories(filter);
    }

    /**
     * Updates a category by ID
     * @param params - Update parameters (id and data)
     * @returns The updated category
     */
    public updateCategoryById = async (
        params: IParamsUpdateCategory
    ): Promise<ICategory> => {
        const oldCategory = await this.repository.findCategoryById(params.id);
        if (!oldCategory) throw new AppError('Category not found', 404);

        const category = new Category({
            ...oldCategory,
            ...params.data,
        });

        if (params.data.name) {
            const exist = await this.repository.findCategoryByName(params.data.name);
            if (exist) throw new AppError('Category with name already exist', 400);
        }

        return await this.repository.updateCategoryById(params.id, category) as ICategory;
    }

    /**
     * Deletes a category by ID
     * @param id - Category ID
     * @returns The deleted category
     */
    public deleteCategoryById = async (
        id: string
    ): Promise<ICategory> => {
        const category = await this.repository.findCategoryById(id);
        if (!category) throw new AppError('Category not found', 404);

        return await this.repository.deleteCategoryById(id) as ICategory;
    }
}