import { Request, Response } from "express";
import { IClassService } from "./interfaces/class.service.interface";
import { IClassSchemas } from "../../main/schemas/class.schema";

/**
 * Controller responsible for handling class-related HTTP requests.
 * Provides endpoints for creating, retrieving, updating, and deleting classs.
 */
export class ClassController {
    private readonly service: IClassService;
    private readonly schemas: IClassSchemas;

    constructor (
        service: IClassService,
        schemas: IClassSchemas,
    ) {
        this.service = service;
        this.schemas = schemas;
    }

    /**
     * Creates a new class
     */
    public createClass = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.schemas.createClass.parse(req);

        const classData = await this.service.createClass(body);

        res.status(201).json({ message: "Class created successfully", class: classData });
    }

    /**
     * Finds a class by ID
     */
    public getClassById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getClassById.parse(req);

        const classData = await this.service.getClassById(params.id);

        res.status(200).json({ class: classData });
    }

    /**
     * Lists all classs
     */
    public listClasses = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { query } = this.schemas.getClasses.parse(req);

        const classs = await this.service.listClasses(query);

        res.status(200).json(classs);
    }

    /**
     * Updates a class by ID
     */
    public updateClass = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body, params } = this.schemas.updateClass.parse(req);

        const classData = await this.service.updateClassById(
            {
                id: params.id,
                data: body,
            }
        );

        res.status(200).json({ message: "Class updated successfully", class: classData });
    }

    /**
     * Deletes a class by ID
     */
    public deleteClass = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.deleteClass.parse(req);

        const classData = await this.service.deleteClassById(params.id);

        res.status(200).json({ message: "Class deleted successfully", class: classData });
    }
}