import { IClass } from "./interfaces/class.interface";

export class Class {
	private data: IClass;

	constructor (
        classData: IClass
    ) {
		Class.validateRequiredFields(classData);

		Class.validateInformation(classData);

        this.data = classData;
	}

    public getData(): IClass {
        return this.data;
    }

    private static validateRequiredFields(
        classData: IClass
    ): void {
        const fields: (keyof IClass)[] = [
            'main_student_id',
            'teacher_id',
            'topic_id',
            'start_time',
            'end_time',
            'type',
            'status',
            'payment_status',
        ];

        for (const field of fields) {
            if (!classData[field]) {
                throw new AppError(`Missing required field: ${field}`, 400);
            }
        }
    }

    private static validateInformation(
        classData: IClass
    ): void {
        const startTime = new Date(classData.start_time);
        const endTime = new Date(classData.end_time);

        if(startTime >= endTime) {
            throw new AppError("Invalid start or end times", 400);
        }

        if(
            ![
                "individual",
                "group"
            ]
            .includes(classData.type)
        ) {
            throw new AppError("The type of class should be individual or group", 400);
        }

        if(
            ![
                "scheduled",
                "canceled",
                "completed"
            ]
            .includes(classData.status)
        ) {
            throw new AppError("Invalid class status", 400);
        }

        if(
            ![
                "pending",
                "paid",
                "refunded"
            ]
            .includes(classData.payment_status)
        ) {
            throw new AppError("Invalid class payment status", 400);
        }
        }
}