import { IClassRepository } from "./interfaces/class.repository.interface";
import { IClass } from "./interfaces/class.interface";
import { Class } from "./class.entity";
import {
    IClassService,
    IParamsCreateClass,
    IClassDependences,
    IParamsUpdateClass,
    IClassFilter,
    IListStudents,
    IGetTeacherById,
    IGetTopicById,
    IGetStudentById
} from "./interfaces/class.service.interface";
import { IStudent } from "../Student/interfaces/student.interface";
import { StreamingService } from "../../shared/services/streaming.service";

/**
 * Service responsible for business logic related to Classes.
 * Handles creation, retrieval, update, and deletion of classes.
 */
export class ClassService implements IClassService {
    private readonly repository: IClassRepository;
    private readonly streamingService: StreamingService;
    public listStudents?: IListStudents;
    public getStudentById?: IGetStudentById;
    public getTeacherById?: IGetTeacherById;
    public getTopicById?: IGetTopicById;

    constructor (
        {
            repository
        }: IClassDependences
    ) {
        this.repository = repository;
        this.streamingService = new StreamingService();
    }

    /**
     * Creates a new class
     * @param classData - Data for class creation
     * @returns The created class
     */
    public createClass = async (
        classData: IParamsCreateClass
    ): Promise<IClass> => {

        // Create a new instance of class
        const newClass = new Class({
            ...classData,
            type: "individual", // inicialize as individual
            end_time: classData.end_time ?? new Date(classData.start_time.getTime() + 50 * 60000),
            name: "", // inicialize without name
            payment_status: "pending", // mock
            status: "scheduled", // on creation, it is always "scheduled"
        });

        // Verify if there is already a class scheduled for that time
        const existsClasses = await this.repository.listClasses({
            start_time: new Date(classData.start_time),
            end_time: new Date(classData.start_time.getTime() + 50 * 60000),
            status: "scheduled",
        });
        if (existsClasses.length > 0) throw new AppError("It is impossible to schedule classes at the same time", 400);

        // Verify if the student exists
        const student = await (this.getStudentById as IGetStudentById)(classData.main_student_id);

        // Verify if the teacher exists
        // It throws an error if it doesn't exist
        await (this.getTeacherById as IGetTeacherById)(classData.teacher_id);

        // Verify if the topic exists
        // It throws an error if it doesn't exist
        const classTopic = await (this.getTopicById as IGetTopicById)(classData.topic_id);

        // Use the getter to get the data
        const createdClass = newClass.getData();

        // Verify the type of class
        if (classData.additional_students && classData.additional_students.length > 0) {

            // Verify the student's data for permission
            this.validateStudentRanking(student);

            // It prevents the student from inviting himself
            const additional_students = classData.additional_students.filter(std => std !== student.id);

            // Valdiate the ID of invited students
            createdClass.additional_students = await this.validateStudentIds(additional_students);
            createdClass.type = "group";
        }

        // Create the class name
        createdClass.name = this.constructClassName(student.full_name, classData.start_time.toString(), classTopic.name);

        // Create the streaming
        const newCall = await this.streamingService.createCall({
            title: createdClass.name,
            description: ``,
            expiresAt: createdClass.end_time,
        });

        // Save the URL and ID in database
        createdClass.streaming_id = newCall.id;
        createdClass.link = newCall.url;

        return await this.repository.createClass(createdClass);
    }

    /**
     * Finds a class by ID
     * @param id - Class ID
     * @returns The found class or throws error if not found
     */
    public getClassById = async (
        id: string
    ): Promise<IClass> => {
        const classData = await this.repository.findClassById(id);

        if (!classData) throw new AppError('Class not found', 404);

        return classData;
    }

    /**
     * Lists classes based on a filter
     * @param filter - The search filter criteria
     * @returns An array of found classes
     */
    public listClasses = async (
        filter: IClassFilter
    ): Promise<IClass[]> => {
        return await this.repository.listClasses(filter);
    }

    /**
     * Updates a class by ID
     * @param params - Update parameters (id and data)
     * @returns The updated class
     */
    public updateClassById = async (
        params: IParamsUpdateClass
    ): Promise<IClass> => {

        // Verify if class exists
        const oldClass = await this.repository.findClassById(params.id);
        if (!oldClass) throw new AppError('Class not found', 404);

        // Validate the times
        if (params.data.start_time) {
            params.data.end_time = params.data.end_time ?? new Date (params.data.start_time?.getTime() + 50 * 60000)
        }

        // Merge the data
        const classData = new Class({
            ...oldClass,
            ...params.data
        });

        const newClass = classData.getData();

        // Verify if the teacher exists
        // It throws an error if it doesn't exist
        if (params.data.teacher_id) {
            await (this.getTeacherById as IGetTeacherById)(params.data.teacher_id);
            newClass.teacher_id = params.data.teacher_id;
        }

        // Verify the type of class
        if (params.data.additional_students && params.data.additional_students.length > 0) {

            // Verify the student's data for permission
            if (newClass.type === 'individual') {
                const student = await (this.getStudentById as IGetStudentById)(newClass.main_student_id);
                this.validateStudentRanking(student);
            }

            // Valdiate the ID of invited students
            newClass.additional_students = await this.validateStudentIds(params.data.additional_students);
        }

        // Recreates the class name
        // Colect the data from the old name
        const oldName = newClass.name.split(' ').filter(a => a !== "-");

        // Inicializes the new name data
        const dataName = {
            start_time: new Date(`${oldName[1].split('/').reverse().join('-')}T${oldName[2]}`).toISOString(),
            topicName: oldName[3],
        }

        // Verify if the topic exists
        // It throws an error if it doesn't exist
        if (params.data.topic_id) {
            const classTopic = await (this.getTopicById as IGetTopicById)(params.data.topic_id);
            newClass.topic_id = `${classTopic.id}`;

            // Saves the topic name
            dataName.topicName = classTopic.name;
        }

        // Saves the new start date
        if (params.data.start_time) {
            dataName.start_time = new Date(params.data.start_time).toISOString();
        }

        if (params.data.status) {
            if (params.data.status !== "scheduled") {

                // Delete the call
                await this.streamingService.deleteCall(newClass.streaming_id as string);

                // Delete the data
                newClass.status = params.data.status;
                newClass.link = "";
                newClass.streaming_id = "";
            }
        }

        // Construct the new class name
        newClass.name = this.constructClassName(oldName[0], dataName.start_time, dataName.topicName);

        return await this.repository.updateClassById(params.id, newClass) as IClass;
    }

    /**
     * Deletes a class by ID
     * @param id - Class ID
     * @returns The deleted class
     */
    public deleteClassById = async (
        id: string
    ): Promise<IClass> => {
        const classData = await this.repository.findClassById(id);

        if (!classData) throw new AppError('Class not found', 404);

        // Delete the streaming
        await this.streamingService.deleteCall(classData.streaming_id as string);

        return await this.repository.deleteClassById(id) as IClass;
    }

    /**
     * Validates if a student is permitted to schedule a group class.
     * @param student - The student's data.
     * @throws Throws an error if the student is not authorized.
     */
    private validateStudentRanking = (
        student: IStudent
    ): void => {
        if (
            student.ranking < 3 || // 0.0 - 5.0 'stars'
            student.total_classes < 5 // min 5 classes
        ) {
            throw new AppError("The student is not authorized to schedule group classes", 400);
        }
    }

    /**
     * Verifies the existence of students with the given IDs.
     * @param IDs - An array of student IDs.
     * @returns An array of valid and found student IDs.
     */
    private validateStudentIds = async (
        IDs: string[]
    ): Promise<string[]> => {
        // Ensures that no more than 3 IDs are saved
        if (IDs.length > 3) throw new AppError("Group calls only accept invitations for up to 3 students", 400);

        const addStudents = await (this.listStudents as IListStudents)({ id: { $in: IDs } });
        const foundIds = addStudents.map((std) => std.id);
        const validRequiredIds = IDs.filter((id) => foundIds.includes(id));

        // Verify if at least one ID is valid
        if (validRequiredIds.length === 0) throw new AppError("Invalid IDs for inviting", 400);

        return validRequiredIds;
    }

    /**
     * Constructs the class name based on provided data
     * @param studentName - The main student's full name
     * @param start_time - The start date and time of the class
     * @param topicName - The name of the class topic
     * @returns The class name in the format "{FirstName} - DD/MM/YYYY HH:MM - {TopicName}"
     */
    private constructClassName = (
        studentName: string,
        start_time: string,
        topicName: string,
    ): string => {
        // Create the name of class
        const firstNameStudent = studentName.split(' ')[0];
        const startTime = new Date(start_time);
        const classDay = startTime.toLocaleDateString("pt-BR");
        const classHour = startTime.toLocaleTimeString("pt-BR", { hour: '2-digit', minute: '2-digit', hour12: false });

        // Construct the name of class
        return `${firstNameStudent} - ${classDay} ${classHour} - ${topicName}`;
    }
}