import { Request, Response } from "express";
import { IDisciplineSchemas } from "../../main/schemas/discipline.schema";
import { IDisciplineService } from "./interfaces/discipline.service.interface";

/**
 * Controller responsible for handling discipline-related HTTP requests.
 * Provides endpoints for creating, retrieving, updating, and deleting disciplines.
 */
export class DisciplineController {
    private readonly service: IDisciplineService;
    private readonly schemas: IDisciplineSchemas;

    constructor (
        service: IDisciplineService,
        schemas: IDisciplineSchemas,
    ) {
        this.service = service;
        this.schemas = schemas;
    }

    /**
     * Creates a new discipline
     */
    public createDiscipline = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.schemas.createDiscipline.parse(req);
        const { name, description, category_id } = body;

        const discipline = await this.service.createDiscipline(
            {
                name,
                description: description || "",
                category_id,
            }
        );

        res.status(201).json({ message: "Discipline created successfully", discipline });
    }

    /**
     * Finds a discipline by ID
     */
    public getDisciplineById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getDisciplineById.parse(req);
        const { id } = params;

        const discipline = await this.service.getDisciplineById(id);

        res.status(200).json(discipline);
    }

    /**
     * Lists all disciplines
     */
    public listDisciplines = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { query } = this.schemas.listDisciplines.parse(req);

        const disciplines = await this.service.listDisciplines(query);

        res.status(200).json(disciplines);
    }

    /**
     * Updates a discipline by ID
     */
    public updateDiscipline = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body, params } = this.schemas.updateDiscipline.parse(req);

        const discipline = await this.service.updateDisciplineById(
            {
                id: params.id,
                data: body
            }
        );

        res.status(200).json({ message: "Discipline updated successfully", discipline });
    }

    /**
     * Deletes a discipline by ID
     */
    public deleteDiscipline = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getDisciplineById.parse(req);

        const discipline = await this.service.deleteDisciplineById(params.id);

        res.status(200).json({ message: "Discipline deleted successfully", discipline });
    }
}