import { IDisciplineRepository } from "./interfaces/discipline.repository.interface";
import { IDiscipline } from "./interfaces/discipline.interface";
import { Discipline } from "./discipline.entity";
import {
    IDisciplineService,
    IGetCategoryById,
    IParamsCreateDiscipline,
    IParamsDisciplineService,
    IParamsUpdateDiscipline
} from "./interfaces/discipline.service.interface";

/**
 * Service responsible for business logic related to Disciplines.
 * Handles creation, retrieval, update, and deletion of disciplines.
 */
export class DisciplineService implements IDisciplineService {
    private readonly repository: IDisciplineRepository;
    public getCategoryById?: IGetCategoryById;

    constructor (
        { repository }: IParamsDisciplineService
    ) {
        this.repository = repository;
    }

    /**
     * Creates a new discipline
     * @param discipline - Data for discipline creation
     * @returns The created discipline
     */
    public createDiscipline = async (
        discipline: IParamsCreateDiscipline
    ): Promise<IDiscipline> => {
        const newDiscipline = new Discipline(discipline);

        const exist = await this.repository.findDisciplineByName(discipline.name);
        if (exist) throw new AppError('Discipline with name already exist', 400);

        const category = await (this.getCategoryById as IGetCategoryById)(discipline.category_id);
        if (!category) throw new AppError('Category with name not found', 404);

        return await this.repository.createDiscipline(newDiscipline);
    }

    /**
     * Finds a discipline by ID
     * @param id - Discipline ID
     * @returns The found discipline or throws error if not found
     */
    public getDisciplineById = async (
        id: string
    ): Promise<IDiscipline> => {
        const discipline = await this.repository.findDisciplineById(id);
        if (!discipline) throw new AppError('Discipline not found', 404);

        return discipline;
    }

    /**
     * Lists disciplines according to filter
     * @param filter - Search filter
     * @returns Array of found disciplines
     */
    public listDisciplines = async (
        filter: Partial<IDiscipline> = {}
    ): Promise<IDiscipline[]> => {

        /*
         * Correctly configure the filter
        */

        return await this.repository.listDisciplines(filter);
    }

    /**
     * Updates a discipline by ID
     * @param params - Update parameters (id and data)
     * @returns The updated discipline
     */
    public updateDisciplineById = async (
        params: IParamsUpdateDiscipline
    ): Promise<IDiscipline> => {
        const oldDiscipline = await this.repository.findDisciplineById(params.id);
        if (!oldDiscipline) throw new AppError('Discipline not found', 404);

        const discipline = new Discipline({
            ...oldDiscipline,
            ...params.data,
        });

        if (params.data.name) {
            const exist = await this.repository.findDisciplineByName(params.data.name);
            if (exist) throw new AppError('Discipline with name already exist', 400);
        }

        if (params.data.category_id) {
            const category = await (this.getCategoryById as IGetCategoryById)(params.data.category_id);
            if (!category) throw new AppError('Category with name not found', 404);
        }

        return await this.repository.updateDisciplineById(params.id, discipline) as IDiscipline;
    }

    /**
     * Deletes a discipline by ID
     * @param id - Discipline ID
     * @returns The deleted discipline
     */
    public deleteDisciplineById = async (
        id: string
    ): Promise<IDiscipline> => {
        const discipline = await this.repository.findDisciplineById(id);
        if (!discipline) throw new AppError('Discipline not found', 404);

        return await this.repository.deleteDisciplineById(id) as IDiscipline;
    }
}