import { Request, Response } from "express";
import { IRoleSchemas } from "../../main/schemas/roles.schema";
import { IRoleService } from "./interfaces/roles.service.interface";

/**
 * Controller responsible for handling Role-related HTTP requests.
 * Provides endpoints for creating, retrieving, updating, and deleting roles.
 */
export class RoleController {
    private readonly roleService: IRoleService;
    private readonly roleSchemas: IRoleSchemas;

    constructor (
        roleService: IRoleService,
        roleSchema: IRoleSchemas,
    ) {
        this.roleService = roleService;
        this.roleSchemas = roleSchema;
    }

    /**
     * Create a new role
     */
    public createRole = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { body } = this.roleSchemas.createRole.parse(req);
        const { name, description } = body;

        const role = await this.roleService.createRole({ name, description }); 

        res.status(201).json({ message: 'Role created successfully', role });
    }

    /**
     * List all roles
     */
    public getRoles = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const roles = await this.roleService.listRoles({});

        res.status(200).json(roles);
    }

    /**
     * Fetch a role by its ID
     */
    public getRoleById = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params } = this.roleSchemas.getRoleById.parse(req);

        const role = await this.roleService.findRoleById(params.id);

        res.status(200).json(role);
    }

    /**
     * Update an existing role
     */
    public updateRole = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { body, params } = this.roleSchemas.updateRole.parse(req);
        const { name, description } = body;

        const role = await this.roleService.updateRole({
            id: params.id,
            data: {
                name,
                description
            }
        });

        res.status(200).json({ message: 'Role updated sucessfully', role });
    }

    /**
     * Delete a role by its ID
     */
    public deleteRole = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params } = this.roleSchemas.deleteRoleById.parse(req);

        const role = await this.roleService.deleteRole(params.id);

        res.status(200).json({ message: 'Role deleted sucessfully', role });
    }
}