import { IRoleRepository } from "./interfaces/roles.repository.interface";
import { IRole } from "./interfaces/roles.interface";
import { Role } from "./roles.entity";
import {
    IRoleService,
    IParamsRoleService,
    IParamsCreateRole,
    IParamsUpdateRole,
} from "./interfaces/roles.service.interface";

/**
 * Service responsible for business logic related to Roles.
 * Handles creation, retrieval, update, and deletion of roles.
 */
export class RoleService implements IRoleService {
    private repository: IRoleRepository;

    constructor (
        { RoleRepository }: IParamsRoleService
    ) {
        this.repository = RoleRepository;
    }

    /**
     * Create a new role
     * @param params - Role data to create
     * @returns The created role object
     */
    public createRole = async (
        params: IParamsCreateRole
    ): Promise<IRole> => {
        const role: IRole = new Role(params);

        const exists = await this.repository.findRoleByName(params.name);
        if (exists) throw new AppError('Name already in use', 400);

        return this.repository.createRole(role);
    }

    /**
     * Find a role by its ID
     * @param id - Role ID
     * @returns The role object or null if not found
     */
    public findRoleById = async (
        id: string,
    ): Promise<IRole> => {
        const role = await this.repository.findRoleById(id);
        if (!role) throw new AppError('Role not found', 404);

        return role;
    }

    /**
     * List all roles with optional filter
     * @param filter - Filter object
     * @returns Array of roles
     */
    public listRoles = async (
        filter: Partial<IRole> = {}
    ): Promise<IRole[]> => {
        return this.repository.listRole(filter);
    }

    /**
     * Update an existing role
     * @param params - Update parameters including role ID and new data
     * @returns The updated role object or null if not found
     */
    public updateRole = async (
        params: IParamsUpdateRole
    ): Promise<IRole> => {
        const role = await this.repository.findRoleById(params.id);
        if (!role) throw new AppError('Role not found', 404);

        const newRole = new Role({
            ...role,
            ...params.data,
        });

        if (params.data.name && params.data.name !== newRole.name) {
            const existing = await this.repository.findRoleByName(params.data.name);
            if (existing) throw new AppError('Name already in use', 400);
        }

        return await this.repository.updateRoleById(
            params.id,
            newRole,
        ) as IRole;
    }

    /**
     * Delete a role by its ID
     * @param id - Role ID
     * @returns The deleted role object or null if not found
     */
    public deleteRole = async (
        id: string
    ): Promise<IRole> => {
        const role = await this.repository.findRoleById(id);
        if (!role) throw new AppError('Role not found', 404);

        return await this.repository.deleteRoleById(id) as IRole;
    }
}