import { IAuthenticatableEntity } from "../../auth/interfaces/auth.interface";
import { IRegisterParams } from "../../auth/interfaces/auth.service.interface";
import { IStudent } from "./student.interface";

/**
 * Interface to adapt Student for authentication
 * Extends IAuthenticatableEntity to ensure compatibility with AuthService
 */
export interface IRegisterStudent extends IRegisterParams {
    full_name: string;
    birth_date: Date;
    cpf: {
        value: string;
        upload_id?: string;
    };
    telephone: string;
    academic: {
        current_grade: string;
        school?: string;
        subjects: string[];
        difficulty_level: string;
        main_goal: string;
    };
    ranking: number;
    total_classes: number;
    class_preferences: {
        modality: 'online' | 'presencial' | 'hibrida';
        frequency: string;
        available_times: Array<{
            day_of_week: number;
            start_time: number;
            end_time: number;
        }>;
        preferred_teacher?: string;
    };
    guardian?: {
        full_name: string;
        birth_date: Date;
        telephone: string;
        email: string;
        relationship: string;
        cpf: {
            value: string;
            upload_id?: string;
        };
        rg: {
            value: string;
            upload_id?: string;
        };
    };
}

/**
 * Interface to adapt Student for authentication
 * Extends IAuthenticatableEntity to ensure compatibility with AuthService
 */
export interface IStudentAuthAdapter extends IAuthenticatableEntity {
    full_name: string;
    birth_date: Date;
    gender?: string;
    cpf: {
        value: string;
        upload_id?: string;
    };
    rg?: {
        value?: string;
        upload_id?: string;
    };
    telephone: string;
    academic: {
        current_grade: string;
        school?: string;
        subjects: string[];
        difficulty_level: string;
        main_goal: string;
    };
    ranking: number;
    total_classes: number;
    class_preferences: {
        modality: 'online' | 'presencial' | 'hibrida';
        frequency: string;
        available_times: Array<{
            day_of_week: number;
            start_time: number;
            end_time: number;
        }>;
        preferred_teacher?: string;
    };
    guardian?: {
        full_name: string;
        birth_date: Date;
        telephone: string;
        email: string;
        relationship: string;
        cpf: {
            value: string;
            upload_id?: string;
        };
        rg: {
            value: string;
            upload_id?: string;
        };
    };
    financial?: {
        credit_card_registered: boolean;
        card_info?: {
            cardholder_name: string;
            card_number: string;
            expiration: string;
            cvv: string;
        };
    };
    password_reset?: {
        token: string | null;
        expires_at: Date | null;
    };
    created_at?: Date;
    updated_at?: Date;
}

/**
 * Function to convert IStudent to IStudentAuthAdapter
 * @param student - Student data
 * @returns Data adapted for authentication
 */
export function adaptStudentForAuth(
    student: IStudent
): IStudentAuthAdapter {
    return {
        id: student.id,
        email: student.email,
        password: student.password,
        full_name: student.full_name,
        birth_date: student.birth_date,
        gender: student.gender,
        cpf: student.cpf,
        rg: student.rg,
        telephone: student.telephone,
        academic: student.academic,
        class_preferences: student.class_preferences,
        ranking: student.ranking,
        total_classes: student.total_classes,
        guardian: student.guardian,
        financial: student.financial,
        password_reset: student.password_reset,
        created_at: student.created_at,
        updated_at: student.updated_at,
    };
}

/**
 * Function to convert IStudentAuthAdapter to IStudent
 * @param authStudent - Authenticated student data
 * @returns Student data
 */
export function adaptAuthStudentToStudent(
    authStudent: IStudentAuthAdapter
): IStudent {
    return {
        id: authStudent.id,
        full_name: authStudent.full_name,
        birth_date: authStudent.birth_date,
        gender: authStudent.gender,
        cpf: authStudent.cpf,
        rg: authStudent.rg,
        email: authStudent.email,
        password: authStudent.password,
        ranking: authStudent.ranking,
        total_classes: authStudent.total_classes,
        telephone: authStudent.telephone,
        academic: authStudent.academic,
        class_preferences: authStudent.class_preferences,
        guardian: authStudent.guardian,
        financial: authStudent.financial,
        password_reset: authStudent.password_reset,
        created_at: authStudent.created_at,
        updated_at: authStudent.updated_at,
    };
}