import { Request, Response } from "express";
import { IStudentService } from "./interfaces/student.service.interface";
import { IStudentSchemas } from "../../main/schemas/student.schema";

/**
 * Controller responsible for handling student-related HTTP requests.
 * Provides endpoints for creating, retrieving, updating, and deleting students.
 */
export class StudentController {
    private readonly service: IStudentService;
    private readonly schemas: IStudentSchemas;

    constructor (
        service: IStudentService,
        schemas: IStudentSchemas,
    ) {
        this.service = service;
        this.schemas = schemas;
    }

    /**
     * Creates a new student
     */
    public createStudent = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.schemas.createStudent.parse(req);

        const student = await this.service.createStudent(body);

        res.status(201).json({ message: "Student created successfully", student });
    }

    /**
     * Finds a student by ID
     */
    public getStudentById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { id } = this.schemas.getStudentById.parse(req).params;

        const student = await this.service.getStudentById(id);

        res.status(200).json(student);
    }

    /**
     * Lists all students
     */
    public listStudents = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const students = await this.service.listStudents();

        res.status(200).json(students);
    }

    /**
     * List students with adaptor
     */
    public listAdaptedStudents = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { query } = this.schemas.listAdaptedStudents.parse(req);

        const students = await this.service.listAdaptedStudents(query);

        res.status(200).json(students);
    }

    /**
     * Updates a student by ID
     */
    public updateStudent = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.schemas.updateStudent.parse(req);

        const student = await this.service.updateStudent(
            {
                // Ensures that the user only updates their own data
                id: req.user?.id as string,
                data: body,
            },
        );

        res.status(200).json({ message: "Student updated successfully", student });
    }

    /**
     * Deletes a student by ID
     */
    public deleteStudent = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { id } = req.params;

        const student = await this.service.deleteStudent(id);

        res.status(200).json({ message: "Student deleted successfully", student });
    }

    /**
     * Find teachers for students
     */
    public getTeachers = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const {
            days_of_week,
            start_time,
            end_time,
            discipline_id,
        } = this.schemas.getTeachers.parse(req).query;

        const teachers = await this.service.getTeachers({
            daysOfWeek: days_of_week,
            startMinutes: start_time,
            endMinutes: end_time,
            disciplineId: discipline_id,
        });

        res.status(200).json({ teachers });
    }

    /**
     * Create scheduled class
     */
    public createClass = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        // ensures that the student schedules classes only for themselves
        req.body.main_student_id = req.user?.id;

        const { body } = this.schemas.createClass.parse(req);

        const createdClass = await this.service.createClass(body);

        res.status(200).json({ message: "Scheduled class successfully", class: createdClass });
    }

    /**
     * List all student's classes
     */
    public getClasses = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        // Ensures that the student only sees their own classes
        req.query.student_id = req.user?.id;
        const { query } = this.schemas.getClasses.parse(req);

        const classes = await this.service.listClasses(query);

        res.status(200).json(classes);
    }

    /**
     * Find one class by ID
     */
    public getClassById = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params } = this.schemas.getClassById.parse(req);

        const classFound = await this.service.getClassById(params.id);

        res.status(200).json({ class: classFound });
    }

    /**
     * Update a class by ID
     */
    public updateClass = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { body, params } = this.schemas.updateClass.parse(req);

        const newClass = await this.service.updateClass({
            id: params.id,
            data: body,
        });

        res.status(200).json({ message: "Class schedule updated successfully", class: newClass });
    }

    /**
     * Cancel class by ID
     */
    public cancelClass = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params } = this.schemas.getClassById.parse(req);

        const canceledClass = await this.service.cancelClass(params.id);

        res.status(200).json({ message: "Class cancelled successfully", class: canceledClass });
    }
}