import { BcryptService } from "../../shared/services/bcrypt.service";
import { IStudentRepository } from "./interfaces/student.repository.interface";
import { IMStudent, IStudent } from "./interfaces/student.interface";
import { Student } from "./student.entity";
import {
    IStudentService,
    IParamsCreateStudent,
    IParamsStudentService,
    IParamsUpdateStudent,
    IListTeachers,
    ICreateClass,
    IListClasses,
    IGetClassById,
    IUpdateClass,
    IFilterStudent,
    IStudentFiltered,
} from "./interfaces/student.service.interface";
import { IFilteredTeacher, IFindTeacherFilter } from "../Teacher/interfaces/teacher.service.interface";
import { IClass } from "../Class/interfaces/class.interface";
import { IParamsCreateClass } from "../Class/interfaces/class.service.interface";
import { FilterQuery, PipelineStage } from "mongoose";

/**
 * Service responsible for business logic related to Teachers.
 * Handles creation, retrieval, update, and deletion of teachers.
 */
export class StudentService implements IStudentService {
    private readonly repository: IStudentRepository;
    private readonly bcryptService: BcryptService;

    constructor (
        { repository }: IParamsStudentService
    ) {
        this.repository = repository;
        this.bcryptService = new BcryptService();
    }

    // Dependences
    public listTeachersDep?: IListTeachers;
    public createClassDep?: ICreateClass;
    public listClassesDep?: IListClasses;
    public getClassByIdDep?: IGetClassById;
    public updateClassDep?: IUpdateClass;

    /**
     * Creates a new student
     * @param student - Data for student creation
     * @returns The created student
     */
    public createStudent = async (
        params: IParamsCreateStudent
    ): Promise<IStudent> => {
        // Create a new instance os Student
        const newStudent = new Student({
            ...params,
            ranking: 0,
            total_classes: 0
        });

        // Verify if exist
        const studentExisting = await this.repository.getStudentByEmail(params.email);
        if (studentExisting) throw new AppError('Email already in use', 400);

        // Get the data
        const studentData = newStudent.getData();

        // Hash the password before save in database
        studentData.password = this.bcryptService.hashPassword(params.password);

        return await this.repository.createStudent(studentData);
    }

    /**
     * Finds a student by ID
     * @param id - Student ID
     * @returns The found student or throws error if not found
     */
    public getStudentById = async (
        id: string
    ): Promise<IStudent> => {
        const student = await this.repository.getStudentById(id);
        if (!student) throw new AppError('Student not found', 404);
        
        return student;
    }

    /**
     * Lists students according to filter
     * @param filter - Search filter
     * @returns Array of found students
     */
    public listStudents = async (
        filter?: object
    ): Promise<IStudent[]> => {
        return await this.repository.listStudents(filter || {});
    }

    /**
     * List students for other students
     * We only search for students with the correct ranking and compatible availability
     * @param filter - Specific search filter
     * @returns Array of found students
     */
    public listAdaptedStudents = async (
        filter: IFilterStudent
    ): Promise<IStudentFiltered[]> => {
        const query = this.buildAvailabilityPipeline(filter);

        const allStudents = await this.repository.listStudents(query);

        const permittedStudents = this.validateRankingOfStudents(allStudents);

        return this.adapterStudent(permittedStudents);
    }

    /**
     * Updates a student by ID
     * @param params - Update parameters (id and data)
     * @returns The updated student
     */
    public updateStudent = async (
        params: IParamsUpdateStudent
    ): Promise<IStudent> => {
        const oldStudent = await this.repository.getStudentById(params.id);
        if (!oldStudent) throw new AppError('Student not found', 404);

        // Performs the merge
        const mergedStudent = this.deepMerge(oldStudent, params);

        // Create new instance of Student
        const student = new Student (mergedStudent);

        // Get the student data
        const studentData = student.getData();

        if (params.data.password) {
            studentData.password = this.bcryptService.hashPassword(params.data.password);
        }

        return await this.repository.updateStudent(params.id, studentData) as IStudent;
    }

    /**
     * Deletes a student by ID
     * @param id - Student ID
     * @returns The deleted student
     */
    public deleteStudent = async (
        id: string
    ): Promise<IStudent> => {
        const student = await this.repository.getStudentById(id);
        if (!student) throw new AppError('Student not found', 404);

        return await this.repository.deleteStudent(id) as IStudent;
    }

    /**
     * Performs deep merging between two documents
     * @param target - Base object to merge into
     * @param source - Object containing new values to merge
     * @returns Merged object
     */
    private deepMerge = <T extends object> (
        target: T,
        source: Partial<T>
    ): T => {
        if (!source) return target;

        const isObject = (obj: unknown): obj is Record<string, unknown> =>
            typeof obj === "object" && obj !== null && !Array.isArray(obj);

        const isArray = (arr: unknown): arr is unknown[] => Array.isArray(arr);

        for (const key of Object.keys(source) as (keyof T)[]) {
            const sourceValue = source[key];
            const targetValue = target[key];

            if (isArray(sourceValue)) {
                const mergedArray = isArray(targetValue)
                    ? [...targetValue, ...sourceValue]
                    : [...sourceValue];

                const uniqueArray = mergedArray.filter(
                    (item, index, self) =>
                        index ===
                        self.findIndex((t) =>
                            isObject(t) && isObject(item)
                                ? JSON.stringify(t) === JSON.stringify(item)
                                : t === item
                        )
                );

                target[key] = uniqueArray as T[typeof key];
            } else if (isObject(sourceValue)) {
                const mergedObject = this.deepMerge(
                    isObject(targetValue)
                        ? { ...targetValue }
                        : ({} as Record<string, unknown>),
                    sourceValue as Record<string, unknown>
                );

                target[key] = mergedObject as T[typeof key];
            } else if (sourceValue !== undefined) {
                target[key] = sourceValue as T[typeof key];
            }
        }

        return target;
    }

    /**
     * Adapter for sending data from student.
     * @param students - Array of students document
     * @returns Adapted array of students
     */
    private adapterStudent = (
        students: IStudent[] | []
    ): IStudentFiltered[] => {
        return students.map(
            student => {
                return {
                    name: student.full_name,
                    cpf: student.cpf.value,
                    birth_date: student.birth_date,
                    available_times: student.class_preferences.available_times,
                };
            }
        );
    }

    /**
     * Validates if a student is permitted to schedule a group class.
     * @param student - The student's data.
     * @returns - Array of students
     */
    private validateRankingOfStudents = (
        students: IStudent[]
    ): IStudent[] => {
        return students.filter(
            student => 
                student.ranking >= 3 &&
                student.total_classes >= 5
        );
    }

    /**
     * Constructs MongoDB aggregation pipeline for availability filtering
     * @param filter - Filter criteria for building the pipeline
     * @returns Array of pipeline stages
     */
    private buildAvailabilityPipeline = (
        filter: IFindTeacherFilter
    ): PipelineStage[] => {
        type AvailabilityMatchCondition = {
            day_of_week?: { $in: number[] };
            start_time?: { $lte: number };
            end_time?: { $gte: number };
        };

        const pipeline: PipelineStage[] = [];
        const matchConditions: FilterQuery<IMStudent> = {};

        // Build Availability Condition
        const availabilityCondition: AvailabilityMatchCondition = {};

        // Filter by days of week
        if (filter.daysOfWeek && filter.daysOfWeek.length > 0) {
            availabilityCondition.day_of_week = { $in: filter.daysOfWeek as number[] };
        }

        // Filter by start time
        if (Number.isInteger(filter.startMinutes)) {
            availabilityCondition.start_time = { $lte: filter.startMinutes as number };
        }

        // Filter by end time
        if (Number.isInteger(filter.endMinutes)) {
            availabilityCondition.end_time = { $gte: filter.endMinutes as number };
        }

        // Apply Availability Condition to $elemMatch
        if (Object.keys(availabilityCondition).length > 0) {
            matchConditions["class_preferences.available_times"] = {
                $elemMatch: availabilityCondition
            };
        }
        
        // Apply pipeline only if *any* filters exist (discipline or availability)
        if (Object.keys(matchConditions).length > 0) {
            pipeline.push({ $match: matchConditions });
        }
        
        return pipeline;
    }

    /**
     * Retrieves teachers based on specified filters
     * @param filter - Criteria for filtering teachers
     * @returns List of teachers matching the filter criteria
     */
    public getTeachers = async (
        filter: IFindTeacherFilter
    ): Promise<IFilteredTeacher[]> => {
        return await (this.listTeachersDep as IListTeachers)(filter);
    }

    public createClass = async (
        params: object
    ): Promise<IClass> => {
        return await (this.createClassDep as ICreateClass)(params as IParamsCreateClass);
    }

    public getClassById = async (
        id: string
    ): Promise<IClass> => {
        return await (this.getClassByIdDep as IGetClassById)(id)
    }

    public listClasses = async (
        filter: object
    ): Promise<IClass[]> => {
        return await (this.listClassesDep as IListClasses)(filter);
    }

    public updateClass = async (
        paramas: {
            id: string,
            data: object
        }
    ): Promise<IClass> => {
        return await (this.updateClassDep as IUpdateClass)(paramas);
    }

    public cancelClass = async (
        id: string
    ): Promise<IClass> => {
        return await (this.updateClassDep as IUpdateClass)(
            {   id,
                data: {
                    status: "canceled"
                }
            }
        );
    }
}