import { IRegisterParams } from "../../auth/interfaces/auth.service.interface";
import { IAuthenticatableEntity } from "../../auth/interfaces/auth.interface";
import { ITeacher } from "./teacher.interface";

/**
 * Interface to adapt Teacher for authentication
 * Extends IAuthenticatableEntity to ensure compatibility with AuthService
 */
export interface IRegisterTeacher extends IRegisterParams {
    full_name: string;
    cpfCnpj: {
        value: string;
        upload_id?: string;
        type: 'CPF' | 'CNPJ';
    };
    rg?: {
        value: string;
        upload_id?: string;
    };
    email: string;
    password: string;
    telephone: string;
    birth_date: Date;
    image_profile?: string;
    address?: {
        street: string;
        number: string;
        complement?: string;
        city: string;
        state: string;
        zip_code: string;
    };
    education?: {
        level: string;
        field: string;
        teaching_experience: string;
        disciplines: string[];
        education_levels: string[];
    };
    availability?: {
        in_person_location: string[];
        available_times: Array<{
            day_of_week: number;
            start_time: number;
            end_time: number;
        }>;
        accepts_emergency_classes: boolean;
    };
    payment?: {
        hourly_rate: number;
        payment_info: {
            type: string;
            key: string;
        };
    };
    preferences?: {
        student_type_preference?: string;
        general_observations?: string;
    };
}

/**
 * Interface to adapt Teacher for authentication
 * Extends IAuthenticatableEntity to ensure compatibility with AuthService
 */
export interface ITeacherAuthAdapter extends IAuthenticatableEntity {
    full_name: string;
    cpfCnpj: {
        value: string;
        upload_id?: string;
        type: 'CPF' | 'CNPJ';
    };
    rg?: {
        value: string;
        upload_id?: string;
    };
    email: string;
    password: string;
    telephone: string;
    birth_date: Date;
    image_profile?: string;
    address?: {
        street: string;
        number: string;
        complement?: string;
        city: string;
        state: string;
        zip_code: string;
    };
    education?: {
        level: string;
        field: string;
        teaching_experience: string;
        disciplines: string[];
        education_levels: string[];
    };
    availability?: {
        in_person_location: string[];
        available_times: Array<{
            day_of_week: number;
            start_time: number;
            end_time: number;
        }>;
        accepts_emergency_classes: boolean;
    };
    payment?: {
        hourly_rate: number;
        payment_info: {
            type: string;
            key: string;
        };
    };
    preferences?: {
        student_type_preference?: string;
        general_observations?: string;
    };
    password_reset?: {
        token: string | null;
        expires_at: Date | null;
    };
    created_at?: Date;
    updated_at?: Date;
}

/**
 * Function to convert ITeacher to ITeacherAuthAdapter
 * @param teacher - Teacher data
 * @returns Data adapted for authentication
 */
export function adaptTeacherForAuth(
    teacher: ITeacher
): ITeacherAuthAdapter {
    return {
        id: teacher.id,
        email: teacher.email,
        password: teacher.password,
        password_reset: teacher.password_reset,
        full_name: teacher.full_name,
        cpfCnpj: teacher.cpfCnpj,
        rg: teacher.rg,
        telephone: teacher.telephone,
        birth_date: teacher.birth_date,
        image_profile: teacher.image_profile,
        address: teacher.address,
        education: teacher.education,
        availability: teacher.availability,
        payment: teacher.payment,
        preferences: teacher.preferences,
        created_at: teacher.created_at,
        updated_at: teacher.updated_at,
    };
}

/**
 * Function to convert ITeacherAuthAdapter to ITeacher
 * @param authTeacher - Authenticated teacher data
 * @returns Teacher data
 */
export function adaptAuthTeacherToTeacher(
    authTeacher: ITeacherAuthAdapter
): ITeacher {
    return {
        id: authTeacher.id,
        full_name: authTeacher.full_name,
        cpfCnpj: authTeacher.cpfCnpj,
        rg: authTeacher.rg,
        email: authTeacher.email,
        password: authTeacher.password,
        telephone: authTeacher.telephone,
        birth_date: authTeacher.birth_date,
        image_profile: authTeacher.image_profile,
        address: authTeacher.address,
        education: authTeacher.education,
        availability: authTeacher.availability,
        payment: authTeacher.payment,
        preferences: authTeacher.preferences,
        password_reset: authTeacher.password_reset,
        created_at: authTeacher.created_at,
        updated_at: authTeacher.updated_at,
    };
}