import { Request, Response } from "express";
import { ITeacherService } from "./interfaces/teacher.service.interface";
import { ITeacherSchemas } from "../../main/schemas/teacher.schema";

/**
 * Controller responsible for handling teacher-related HTTP requests.
 * Provides endpoints for creating, retrieving, updating, and deleting teachers.
 */
export class TeacherController {
    private readonly service: ITeacherService;
    private readonly schemas: ITeacherSchemas;

    constructor (
        service: ITeacherService,
        schemas: ITeacherSchemas,
    ) {
        this.service = service;
        this.schemas = schemas;
    }

    /**
     * Create a new teacher
     */
    public createTeacher = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { body } = this.schemas.createTeacher.parse(req);

        const newTeacher = await this.service.createTeacher(body);

        res.status(201).json({ message: "Teacher created successfully", newTeacher });
    }

    /**
     * Fetch a teacher by ID
     */
    public getTeacherById = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params } = this.schemas.getTeacherById.parse(req);

        const teacher = await this.service.getTeacherById(params.id);

        res.status(200).json(teacher);
    }

    /**
     * List all teachers
     */
    public listTeachers = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const teachers = await this.service.listTeacher();

        res.status(200).json(teachers);
    }

    /**
     * Update a teacher by ID
     */
    public updateTeacher = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params, body } = this.schemas.updateTeacher.parse(req);

        const teacher = await this.service.updateTeacherById(
            {
                id: params.id,
                data: body,
            }
        );

        res.status(200).json({ message: "Teacher updated successfully", teacher });
    }

    /**
     * Delete a teacher by ID
     */
    public deleteTeacher = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params } = this.schemas.getTeacherById.parse(req);

        const teacher = await this.service.deleteTeacherById(params.id);

        res.status(200).json({ message: "Teacher deleted successfully", teacher })
    }

    /**
     * Get all the teacher's classes
     */
    public getClasses = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        req.query.teacher_id = req.user?.id; // apply the id in query
        const { query } = this.schemas.getClasses.parse(req);

        const classes = await this.service.listClasses(query);

        res.status(200).json(classes);
    }

    /**
     * Get one class by ID
     */
    public getClassById = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params } = this.schemas.getClassById.parse(req);

        const classFound = await this.service.getClassById(params.id);

        res.status(200).json({ class: classFound });
    }

    /**
     * Finish a class by ID
     */
    public finishClass = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { params } = this.schemas.getClassById.parse(req);

        const classFinished = await this.service.finishClass(params.id);

        res.status(200).json({ message: "Class completed successfully", class: classFinished });
    }
}