import { PipelineStage, FilterQuery } from "mongoose";
import { ITeacherRepository } from "./interfaces/teacher.repository.interface";
import {
    IParamsCreateTeacher,
    IParamsTeacherService,
    IParamsUpdateTeacher,
    ITeacherService,
    IFilteredTeacher,
    IFindTeacherFilter,
    IListTopics,
    IListClasses,
    IGetClassById,
    IClassFilter,
    IUpdateClassById
} from "./interfaces/teacher.service.interface";
import { IMTeacher, ITeacher } from "./interfaces/teacher.interface";
import { Teacher } from "./teacher.entity";
import { BcryptService } from "../../shared/services/bcrypt.service";
import { IClass } from "../Class/interfaces/class.interface";

/**
 * Service responsible for business logic related to Teachers.
 * Handles creation, retrieval, update, and deletion of teachers.
 */
export class TeacherService implements ITeacherService {
    private readonly repository: ITeacherRepository;
    private readonly bcryptService: BcryptService;
    
    constructor (
        {
            repository
        }: IParamsTeacherService
    ) {
        this.repository = repository;
        this.bcryptService = new BcryptService();
    }

    public listTopicsDep?: IListTopics;
    public listClassesDep?: IListClasses;
    public getClassByIdDep?: IGetClassById;
    public updateClassByIdDep?: IUpdateClassById;

    /**
     * Create a new teacher
     * @param params - Teacher data to create
     * @returns The created teacher object
     */
    public createTeacher = async (
        params: IParamsCreateTeacher
    ): Promise<ITeacher> => {
        const teacher = new Teacher(params);

        const exists = await this.repository.getTeacherByEmail(params.email);
        if (exists) throw new AppError('Email already in use', 400);

        const teacherData = teacher.getData();

        // Validate the topics IDs
        if (teacherData.education && teacherData.education.disciplines.length > 0) {

            teacherData.education.disciplines = await this.validateTopicsIds(teacherData.education.disciplines);
        }

        // Hash the password
        teacherData.password = this.bcryptService.hashPassword(params.password)

        return await this.repository.createTeacher(teacherData);
    }

    /**
     * Find a teacher by its ID
     * @param id - Teacher ID
     * @returns The teacher object or null if not found
     */
    public getTeacherById = async (
        id: string,
    ): Promise<ITeacher> => {
        const teacher = await this.repository.getTeacherById(id);
        if (!teacher) throw new AppError('Teacher not found', 401);

        return teacher;
    }

    /**
     * List all teachers with optional filter
     * @param filter - Filter object
     * @returns Array of teachers
     */
    public listTeacher = async (
        filter: object = {}
    ): Promise<ITeacher[]> => {
        return await this.repository.listTeachers(filter);
    }

    /**
     * Update an existing teacher
     * @param params - Update parameters including teacher ID and new data
     * @returns The updated teacher object or null if not found
     */
    public updateTeacherById = async (
        params: IParamsUpdateTeacher
    ): Promise<ITeacher> => {
        const oldTeacher = await this.repository.getTeacherById(params.id);
        if (!oldTeacher) throw new AppError('Teacher not found', 401);

        // Remove null data
        const data = Object.fromEntries(
            Object.entries(params.data).filter(([_, value]) => value != null)
        );
        if (Object.keys(data).length === 0) throw new AppError('Data not provided', 400);

        const teacherData = this.deepMerge<ITeacher>(oldTeacher, data);

        // Create new instance with old data, and overlaying new data
        const newTeacher = new Teacher(teacherData);

        const teacher = newTeacher.getData();

        if (
            (params.data.education && params.data.education.disciplines.length > 0) &&
            (teacher.education && teacher.education.disciplines.length > 0)
        ) {
            teacher.education.disciplines = await this.validateTopicsIds(teacher.education.disciplines);
        }

        // Hash the password
        if (params.data.password) {
            teacher.password = this.bcryptService.hashPassword(teacher.password);
        }

        return await this.repository.updateTeacherById(
            params.id, teacher
        ) as ITeacher;
    }

    /**
     * Delete a teacher by its ID
     * @param id - Teacher ID
     * @returns The deleted teacher object or null if not found
     */
    public deleteTeacherById = async (
        id: string
    ): Promise<ITeacher> => {
        const teacher = await this.repository.getTeacherById(id);
        if (!teacher) throw new AppError('Teacher not found', 401);

        return await this.repository.deleteTeacherById(id) as ITeacher;
    }

    /**
     * Validates topic IDs before assigning to teacher
     * @param topics - Array of topic IDs to validate
     * @returns Array of valid topic IDs
     * @throws AppError when no valid topics are found
     */
    private validateTopicsIds = async (
        topics: string[]
    ): Promise<string[]> => {
        const existingTopics = await (this.listTopicsDep as IListTopics)({ 
            _id: { $in: topics } 
        });
        const validIds = existingTopics.map(topic => topic.id);

        if (validIds.length === 0) {
            throw new AppError('Invalid topic IDs', 400);
        }

        return validIds.filter(Boolean) as string[];
    }

    /**
     * Performs deep merging between two documents
     * @param target - Base object to merge into
     * @param source - Object containing new values to merge
     * @returns Merged object
     */
    private deepMerge = <T extends object> (
        target: T,
        source: Partial<T>
    ): T => {
        if (!source) return target;

        const isObject = (obj: unknown): obj is Record<string, unknown> =>
            typeof obj === "object" && obj !== null && !Array.isArray(obj);

        const isArray = (arr: unknown): arr is unknown[] => Array.isArray(arr);

        for (const key of Object.keys(source) as (keyof T)[]) {
            const sourceValue = source[key];
            const targetValue = target[key];

            if (isArray(sourceValue)) {
                const mergedArray = isArray(targetValue)
                    ? [...targetValue, ...sourceValue]
                    : [...sourceValue];

                const uniqueArray = mergedArray.filter(
                    (item, index, self) =>
                        index ===
                        self.findIndex((t) =>
                            isObject(t) && isObject(item)
                                ? JSON.stringify(t) === JSON.stringify(item)
                                : t === item
                        )
                );

                target[key] = uniqueArray as T[typeof key];
            } else if (isObject(sourceValue)) {
                const mergedObject = this.deepMerge(
                    isObject(targetValue)
                        ? { ...targetValue }
                        : ({} as Record<string, unknown>),
                    sourceValue as Record<string, unknown>
                );

                target[key] = mergedObject as T[typeof key];
            } else if (sourceValue !== undefined) {
                target[key] = sourceValue as T[typeof key];
            }
        }

        return target;
    }

    /**
     * Retrieves teachers based on specified filters
     * @param filter - Criteria for filtering teachers
     * @returns List of teachers matching the filter criteria
     */
    public findTeacherForStudent = async (
        filter: IFindTeacherFilter
    ): Promise<IFilteredTeacher[]> => {

        // Validate filter parameters
        await this.validateFilter(filter);

        // Construct aggregation pipeline based on filter
        const pipeline = this.buildAvailabilityPipeline(filter);

        if (pipeline.length === 0) {
            // If filters are not passed
            const teachers = await this.repository.listTeachers({});
            return this.adapterTeacherForStudent(teachers);
        } else {
            // If passed
            const teachers = await this.repository.findTeachersWithAggregate(pipeline);
            return this.adapterTeacherForStudent(teachers);
        }
    }

    /**
     * Constructs MongoDB aggregation pipeline for availability filtering
     * @param filter - Filter criteria for building the pipeline
     * @returns Array of pipeline stages
     */
    private buildAvailabilityPipeline = (
        filter: IFindTeacherFilter
    ): PipelineStage[] => {
        type AvailabilityMatchCondition = {
            day_of_week?: { $in: number[] };
            start_time?: { $lte: number };
            end_time?: { $gte: number };
        };

        const pipeline: PipelineStage[] = [];
        const matchConditions: FilterQuery<IMTeacher> = {};
        
        // Filter by discipline
        if (
            typeof filter.disciplineId === 'string' &&
            filter.disciplineId !== '' &&
            filter.disciplineId !== "undefined"
        ) {
            matchConditions["education.disciplines"] = filter.disciplineId;
        }
        
        // Build Availability Condition
        const availabilityCondition: AvailabilityMatchCondition = {};

        // Filter by days of week
        if (filter.daysOfWeek && filter.daysOfWeek.length > 0) {
            availabilityCondition.day_of_week = { $in: filter.daysOfWeek as number[] };
        }

        // Filter by start time
        if (Number.isInteger(filter.startMinutes)) {
            availabilityCondition.start_time = { $lte: filter.startMinutes as number };
        }

        // Filter by end time
        if (Number.isInteger(filter.endMinutes)) {
            availabilityCondition.end_time = { $gte: filter.endMinutes as number };
        }

        // Apply Availability Condition to $elemMatch
        if (Object.keys(availabilityCondition).length > 0) {
            matchConditions["avaliability.available_times"] = {
                $elemMatch: availabilityCondition
            };
        }
        
        // Apply pipeline only if *any* filters exist (discipline or availability)
        if (Object.keys(matchConditions).length > 0) {
            pipeline.push({ $match: matchConditions });
        }
        
        return pipeline;
    }

    /**
     * Validates filter parameters for teacher search
     * @param filter - Filter object to validate
     * @throws AppError when validation fails
     */
    private validateFilter = async (
        filter: IFindTeacherFilter
    ): Promise<void> => {

        if (filter.daysOfWeek !== undefined && filter.daysOfWeek !== null) {

            // Convert to array if single value
            if (!Array.isArray(filter.daysOfWeek)) {
                filter.daysOfWeek = [filter.daysOfWeek];
            }

            // Convert string values to numbers
            filter.daysOfWeek = filter.daysOfWeek.map(day => Number(day));

            // Validate day of week values
            for (const day of filter.daysOfWeek) {
                if (
                    day < 0 ||
                    day > 6 ||
                    isNaN(day)
                ) {
                    throw new AppError('Invalid week days', 400);
                }
            }
        }

        let startTime: number | undefined = undefined;
        let endTime: number | undefined = undefined;

        // Validate the start time
        if (Number.isInteger(filter.startMinutes)) {
            startTime = Number(filter.startMinutes);
            if (!Number.isInteger(startTime)) {
                throw new AppError('Invalid start time, must be an integer', 400);
            }
            filter.startMinutes = startTime;
        } else {
            filter.startMinutes = undefined;
        }

        // Validate the end time
        if (Number.isInteger(filter.endMinutes)) {
            endTime = Number(filter.endMinutes);
            if (!Number.isInteger(endTime)) {
                throw new AppError('Invalid end time, must be an integer', 400);
            }
            filter.endMinutes = endTime;
        } else {
            filter.endMinutes = undefined;
        }

        if (startTime && endTime) {
            if (startTime >= endTime) {
                throw new AppError('Invalid time range: start time must be less than end time', 400);
            }
        }

        // Validate discipline ID
        if (
            typeof filter.disciplineId === 'string' &&
            filter.disciplineId !== "undefined" &&
            filter.disciplineId !== ""
        ) {
            await this.validateTopicsIds([filter.disciplineId]);
        }
    }

    /**
     * Adapter for sending data from teacher to student.
     * @param teachers - Array of teacher document
     * @returns Adapted array of teacher
     */
    private adapterTeacherForStudent = async (
        teachers: ITeacher[] | []
    ): Promise<IFilteredTeacher[]> => {
        return await Promise.all(
            teachers.map(
                async (teacher): Promise<IFilteredTeacher> => {
                    return {
                        id: teacher.id as string,
                        full_name: teacher.full_name,
                        email: teacher.email,
                        telephone: teacher.telephone,
                        image_profile: teacher.image_profile,
                        disciplines: teacher.education?.disciplines ?? [],
                        level_education: teacher.education?.level ?? "",
                        available_times: teacher.availability?.available_times ?? [],
                        classes: await this.listClasses({
                            status: 'scheduled',
                            teacher_id: teacher.id as string,
                        }),
                    };
                }
            )
        );
    }

    public listClasses = async (
        filter: IClassFilter
    ): Promise<IClass[]> => {
        return await (this.listClassesDep as IListClasses)(filter);
    }

    public getClassById = async (
        id: string
    ): Promise<IClass> => {
        return await (this.getClassByIdDep as IGetClassById)(id);
    }

    public finishClass = async (
        id: string
    ): Promise<IClass> => {
        return await (this.updateClassByIdDep as IUpdateClassById)(
            {
                id,
                data: {
                    status: "completed"
                },
            }
        );
    }
}