import { Request, Response } from "express";
import { ITopicSchemas } from "../../main/schemas/topic.schema";
import { ITopicService } from "./interfaces/topic.service.interface";

/**
 * Controller responsible for handling topic-related HTTP requests.
 * Provides endpoints for creating, retrieving, updating, and deleting topics.
 */
export class TopicController {
    private readonly service: ITopicService;
    private readonly schemas: ITopicSchemas;

    constructor (
        service: ITopicService,
        schemas: ITopicSchemas,
    ) {
        this.service = service;
        this.schemas = schemas;
    }

    /**
     * Creates a new topic
     */
    public createTopic = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.schemas.createTopic.parse(req);
        const { name, description, level, discipline_id } = body;

        const topic = await this.service.createTopic(
            {
                name,
                description: description || "",
                level,
                discipline_id,
            }
        );

        res.status(201).json({ message: "Topic created successfully", topic });
    }

    /**
     * Finds a topic by ID
     */
    public getTopicById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getTopic.parse(req);

        const topic = await this.service.getTopicById(params.id);

        res.status(200).json(topic);
    }

    /**
     * Lists all topics
     */
    public listTopics = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { query } = this.schemas.listTopics.parse(req);

        const topics = await this.service.listTopics(query);

        res.status(200).json(topics);
    }

    /**
     * Updates a topic by ID
     */
    public updateTopic = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params, body } = this.schemas.updateTopic.parse(req);

        const topic = await this.service.updateTopicById({ 
            id: params.id, 
            data: body 
        });

        res.status(200).json({ message: "Topic updated successfully", topic });
    }

    /**
     * Deletes a topic by ID
     */
    public deleteTopic = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.deleteTopic.parse(req);

        const topic = await this.service.deleteTopicById(params.id);

        res.status(200).json({ message: "Topic deleted successfully", topic });
    }
}