import { ITopicRepository } from "./interfaces/topic.repository.interface";
import { ITopic } from "./interfaces/topic.interface";
import { Topic } from "./topic.entity";
import {
    ITopicService,
    IParamsCreateTopic,
    IParamsTopicService,
    IParamsUpdateTopic,
    IGetDisciplineById
} from "./interfaces/topic.service.interface";

/**
 * Service responsible for business logic related to Topics.
 * Handles creation, retrieval, update, and deletion of topics.
 */
export class TopicService implements ITopicService {
    private readonly repository: ITopicRepository;
    public getDisciplineById?: IGetDisciplineById;

    constructor (
        { repository }: IParamsTopicService
    ) {
        this.repository = repository;
    }

    /**
     * Creates a new topic
     * @param topic - Data for topic creation
     * @returns The created topic
     */
    public createTopic = async (
        topic: IParamsCreateTopic
    ): Promise<ITopic> => {
        const newTopic = new Topic(topic);

        const exist = await this.repository.findTopicByName(topic.name);
        if (exist) throw new AppError('Topic with name already exist', 400);

        const discipline = await (this.getDisciplineById as IGetDisciplineById)(topic.discipline_id);
        if (!discipline) throw new AppError('Discipline invalid', 400);

        return await this.repository.createTopic(newTopic);
    }

    /**
     * Finds a topic by ID
     * @param id - Topic ID
     * @returns The found topic or throws error if not found
     */
    public getTopicById = async (
        id: string
    ): Promise<ITopic> => {
        const topic = await this.repository.findTopicById(id);

        if (!topic) throw new AppError('Topic not found', 404);

        return topic;
    }

    /**
     * Lists topics according to filter
     * @param filter - Search filter
     * @returns Array of found topics
     */
    public listTopics = async (
        filter?: object
    ): Promise<ITopic[]> => {

        /*
         * Correctly configure the filter
        */

        return await this.repository.listTopics(filter || {});
    }

    /**
     * Updates a topic by ID
     * @param params - Update parameters (id and data)
     * @returns The updated topic
     */
    public updateTopicById = async (
        params: IParamsUpdateTopic
    ): Promise<ITopic> => {
        const oldTopic = await this.repository.findTopicById(params.id);
        if (!oldTopic) throw new AppError('Topic not found', 404);

        const topic = new Topic({
            ...oldTopic,
            ...params.data,
        });

        if (params.data.name) {
            const exist = await this.repository.findTopicByName(params.data.name);
            if (exist) throw new AppError('Topic with name already exist', 400);
        }

        if (params.data.discipline_id) {
            const discipline = await (this.getDisciplineById as IGetDisciplineById)(params.data.discipline_id);
            if (!discipline) throw new AppError('Discipline invalid', 400);
        }

        return await this.repository.updateTopicById(params.id, topic) as ITopic;
    }

    /**
     * Deletes a topic by ID
     * @param id - Topic ID
     * @returns The deleted topic
     */
    public deleteTopicById = async (
        id: string
    ): Promise<ITopic> => {
        const topic = await this.repository.findTopicById(id);

        if (!topic) throw new AppError('Topic not found', 404);

        return await this.repository.deleteTopicById(id) as ITopic;
    }
}