import { Request, Response } from 'express';
import { AuthService } from './auth.service';
import { IAuthenticatableEntity } from './interfaces/auth.interface';
import { IRegisterParams } from './interfaces/auth.service.interface';
import { IAuthSchemas, IUpdateEntity } from '../../main/schemas/auth.schema';

/**
 * Generic authentication controller that can work with any authenticatable entity
 * @template T - Type of entity that extends IAuthenticatableEntity
 * @template R - Type of registration parameters that extends IRegisterParams
 */
export class AuthController<
    T extends IAuthenticatableEntity,
    R extends IRegisterParams
> {
    private readonly authService: AuthService<T, R>;
    private readonly authSchemas: IAuthSchemas;

    constructor (
        authService: AuthService<T, R>,
        authSchemas: IAuthSchemas,
    ) {
        this.authService = authService;
        this.authSchemas = authSchemas;
    }

    /**
     * Registers a new entity
     */
    public register = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.authSchemas.register.parse(req);

        const content = await this.authService.register(body as R);

        res.status(201).json({ message: 'Registered successfully', content });
    };

    /**
     * Logs in an entity
     */
    public login = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.authSchemas.login.parse(req);

        const token = await this.authService.login(body);

        res.status(200).json({ message: 'Login successfully', token });
    };

    /**
     * View profile
     */
    public viewProfile = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const user = await this.authService.getProfile(`${req.user?.id}`);

        res.status(200).json(user);
    };

    /**
     * Edit profile
     */
    public editProfile = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.authSchemas.update.parse(req);
        const id = `${req.user?.id}`;

        const user = await this.authService.updateProfile({
            id,
            data: body as IUpdateEntity,
        })

        res.status(200).json({ message: 'Profile updated successfully', user });
    };

    /**
     * Requests password reset
     */
    public forgotPassword = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.authSchemas.forgotPassword.parse(req);

        const user = await this.authService.forgotPassword(body.email);

        res.status(200).json({
            message: 'Password reset code sent to email',
            expires_at: user.password_reset?.expires_at || '',
        });
    };

    /**
     * Resets password using the code sent by email
     */
    public resetPassword = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.authSchemas.resetPassword.parse(req);
        const { code, email, newPassword } = body;

        const user = await this.authService.resetPassword(
            code,
            email,
            newPassword
        );

        res.status(200).json({ message: 'Password reset successfully', user });
    };
}