import { IAuthJwtPayload, IAuthenticatableEntity } from "./interfaces/auth.interface";
import { IAuthRepository } from "./interfaces/auth.repository.interface";
import { AuthEntityBuilder } from "./interfaces/auth.service.interface";
import { BcryptService } from "../../shared/services/bcrypt.service";
import { JwtService } from "../../shared/services/jwt.service";
import { MailtrapService } from "../../shared/services/mailtrap.service";
import { generateCode } from "../../shared/utils/codeGenerator";
import { 
    IAuthService, 
    IAuthServiceParams, 
    ILoginParams, 
    IRegisterParams,
} from "./interfaces/auth.service.interface";
import { IUpdateEntity } from "../../main/schemas/auth.schema";

type generateCode = () => string;

/**
 * Generic authentication service that can work with any authenticatable entity
 * @template T - Type of entity that extends IAuthenticatableEntity
 * @template R - Type of registration parameters that extends IRegisterParams
 */
export class AuthService<
    T extends IAuthenticatableEntity,
    R extends IRegisterParams,
> implements IAuthService<T, R> {
    private readonly repository: IAuthRepository<T>;
    private readonly Entity: AuthEntityBuilder<T, R>;
    private readonly bcryptService: BcryptService;
    private readonly jwtService: JwtService;
    private readonly mailtrapService: MailtrapService;
    private readonly generateCode: generateCode;

    constructor (
        { repository, entity }: IAuthServiceParams<T, R>
    ) {
        this.repository = repository;
        this.Entity = entity;
        this.bcryptService = new BcryptService();
        this.jwtService = new JwtService();
        this.mailtrapService = new MailtrapService();
        this.generateCode = generateCode;
    }

    /**
     * Register a new user
     * @param params - The user data to register
     */
    public async register(
        params: R,
    ): Promise<T> {
        const newEntity = this.Entity(params);

        const existingUser = await this.repository.findByEmail(params.email);
        if (existingUser) throw new AppError('Email already in use', 400);

        newEntity.password = this.bcryptService.hashPassword(params.password);

        return await this.repository.create(newEntity);
    }

    /**
     * Login a user
     * @param params - The user data to login
     * @returns The JWT token
     */
    public async login(
        params: ILoginParams
    ): Promise<string> {
        if (!params.email || !params.password) throw new AppError('Email and password are required', 400);

        const user = await this.repository.findByEmail(params.email);

        if (!user) throw new AppError('Incorrect email or password', 401);

        const isValidPassword = this.bcryptService.validatePassword(params.password, user.password);

        if (!isValidPassword) throw new AppError('Incorrect email or password', 401);

        // Generate and return JWT token
        const payload: IAuthJwtPayload = {
            id: user.id as string,
            email: user.email,
        };

        return this.jwtService.generateToken(payload);
    }

    public async getProfile(
        id: string,
    ): Promise<T> {
        const user = await this.repository.findById(id);

        if (!user) throw new AppError('User not found', 404);

        return user;
    }

    public async updateProfile(
        params: {
            id: string,
            data: IUpdateEntity,
        },
    ): Promise<T> {
        const oldUser = await this.repository.findById(params.id);

        if (!oldUser) throw new AppError('User not found', 404);

        const newData = {
            ...oldUser,
            ...params.data,
        } as T;

        const newUser = await this.repository.update(params.id, newData);

        if (!newUser) throw new AppError('User could not be updated', 500);

        return newUser;
    }

    /**
     * Verify a JWT token and return the associated user
     * Used in middleware
     * @param token - The JWT token to verify
     */
    public async verifyToken(
        token: string
    ): Promise<IAuthJwtPayload | null> {

        // Verify the token and extract the payload
        const payload = this.jwtService.verifyToken(token) as { id: string; email: string } | null;
        if (!payload) return null;

        // Confirm the user still exists
        const user = await this.repository.findById(payload.id);
        if (!user) return null;

        return payload;
    }

    /**
     * Sends the password reset link to the registered email
     * @param email - User email
     */
    public async forgotPassword(
        email: string
    ): Promise<T> {
        if (!email) throw new AppError('Email are required', 400);

        const user = await this.repository.findByEmail(email);

        if (!user) throw new AppError('User not found', 404);

        // Generate the code and calculate the expires time
        const code = this.generateCode();
        const now = new Date()
        const expiresAt = new Date(now.getTime() + 30 * 60000); // 30m

        const userWithCode = {
            ...user,
            password_reset: {
                token: code,
                expires_at: expiresAt,
            }
        } as Partial<T>;

        // Save token and espires at in database
        const newUseModel = await this.repository.update(user.id as string, userWithCode);

        // Send the email
        await this.mailtrapService.sendOneMail({
            recipient: user.email,
            subject: "Reset your password with the code.",
            text: `Your code is: ${code}`,
        });

        return newUseModel as T;
    }

    /**
     * Reset user password using verification code
     * @param code - The code sent to the user
     * @param email - The user's email
     * @param newPassword - The new password
     */
    public async resetPassword(
        code: string,
        email: string,
        newPassword: string,
    ): Promise<T> {
        // Validate required parameters
        if (!email || !code || !newPassword) {
            throw new AppError('Email, verification code, and new password are required', 400);
        }

        // Verify if the email belongs to an existing user
        const user = await this.repository.findByEmail(email);
        if (!user) {
            throw new AppError('Invalid or expired password reset token', 401);
        }

        // Validate code is correct
        if (user.password_reset?.token != code) {
            throw new AppError('Invalid or expired password reset token', 401);
        }

        // Verify if code not expires
        const now = new Date();
        if (now > (user.password_reset?.expires_at as Date)) {
            throw new AppError('Invalid or expired password reset token', 401);
        }

        // Hash new password
        const hashedPassword = this.bcryptService.hashPassword(newPassword);

        // Save in database
        return await this.repository.update(
            user.id as string,
            {
                ...user,
                password: hashedPassword,
                password_reset: null,
            }
        ) as T;
    }
}