import { IAuthRepository } from "./auth.repository.interface";
import { IAuthenticatableEntity, IAuthJwtPayload } from "./auth.interface";
import { IUpdateEntity } from "../../../main/schemas/auth.schema";

/**
 * Interface for authentication parameters
 */
export interface ILoginParams {
    email: string;
    password: string;
}

/**
 * Interface for registration data
 */
export interface IRegisterParams extends ILoginParams {
    name?: string;
}

/**
 * Generic interface for authentication services
 * @template T - Type of entity that extends IAuthenticatableEntity
 * @template R - Type of registration parameters that extends IRegisterParams
 */
export interface IAuthService<
    T extends IAuthenticatableEntity,
    R extends IRegisterParams
> {
    register (
        params: R
    ): Promise<T>;

    login (
        params: ILoginParams
    ): Promise<string>;

    getProfile(
        id: string
    ): Promise<T>;

    updateProfile(
        params: {
            id: string;
            data: IUpdateEntity;
        }
    ): Promise<T>;

    verifyToken (
        token: string
    ): Promise<IAuthJwtPayload | null>;

    forgotPassword (
        email: string
    ): Promise<T>;

    resetPassword (
        code: string,
        email: string,
        newPassword: string
    ): Promise<T>;
}

/**
 * Builder to validate input data and their types
 */
export type AuthEntityBuilder<
    T extends IAuthenticatableEntity,
    R extends IRegisterParams,
> = (
    params: R
) => T;

/**
 * Interface for AuthService dependency injection parameters
 * @template T - Type of entity that extends IAuthenticatableEntity
 * @template R - Type of registration parameters that extends IRegisterParams
 */
export interface IAuthServiceParams<
    T extends IAuthenticatableEntity,
    R extends IRegisterParams
> {
    repository: IAuthRepository<T>;
    entity: AuthEntityBuilder<T, R>;
}