import { IAuthenticatableEntity } from "../../auth/interfaces/auth.interface";
import { IUser } from "./user.interface";

/**
 * Interface to adapt User for authentication
 * Extends IAuthenticatableEntity to ensure compatibility with AuthService
 */
export interface IUserAuthAdapter extends IAuthenticatableEntity {
    name: string;
    email_verified_at?: Date | null;
    is_active?: boolean;
    roles: string[];
    password_reset?: {
        token: string | null;
        expires_at: Date | null;
    };
    created_at?: Date;
    updated_at?: Date;
}

/**
 * Function to convert IUser to IUserAuthAdapter
 * @param user - User data
 * @returns Data adapted for authentication
 */
export function adaptUserForAuth(
    user: IUser
): IUserAuthAdapter {
    return {
        id: user.id,
        name: user.name,
        email: user.email,
        password: user.password,
        password_reset: user.password_reset,
        email_verified_at: user.email_verified_at,
        is_active: user.is_active,
        roles: user.roles || [],
        created_at: user.created_at,
        updated_at: user.updated_at,
    };
}

/**
 * Function to convert IUserAuthAdapter to IUser
 * @param authUser - Authenticated user data
 * @returns User data
 */
export function adaptAuthUserToUser(
    authUser: IUserAuthAdapter
): IUser {
    return {
        id: authUser.id,
        name: authUser.name,
        email: authUser.email,
        password: authUser.password,
        password_reset: authUser.password_reset,
        is_active: authUser.is_active,
        roles: authUser.roles,
        email_verified_at: authUser.email_verified_at,
        created_at: authUser.created_at,
        updated_at: authUser.updated_at,
    };
}