import { Request, Response } from 'express';
import { IUserService } from './interfaces/user.service.interface';
import { IUserSchemas } from '../../main/schemas/user.schema';

/**
 * Controller responsible for handling user-related HTTP requests.
 * Provides endpoints for creating, retrieving, updating, and deleting users.
 */
export class UserController {
    private readonly service: IUserService;
    private readonly schemas: IUserSchemas;

    constructor (
        service: IUserService,
        schemas: IUserSchemas,
    ) {
        this.service = service;
        this.schemas = schemas;
    }

    /**
     * Fetch all users
     */
    public getUsers = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const users = await this.service.listUsers({});

        res.status(200).json(users);
    };

    /**
     * Fetch a user by ID
     */
    public getUserById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getUserById.parse(req);

        const user = await this.service.getUserById(params.id);

        res.status(200).json(user);
    };

    /**
     * Get the user roles
     */
    public getUserRolesById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getUserRolesById.parse(req);

        const userRoles = await this.service.getRolesFromUserById(params.id);

        res.status(200).json(userRoles);
    }

    /**
     * Create a new user
     */
    public createUser = async (
        req: Request,
        res: Response
    ): Promise<void> => {
        const { body } = this.schemas.createUser.parse(req);
        const { name, email, password } = body;

        const user = await this.service.createUser({
            name,
            email,
            password,
        });

        res.status(201).json({ message: "User created successfully", user });
    };

    /**
     * Assign one or more role to a user
     */
    public assignRoleToUser = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params, body } = this.schemas.assignRoleToUser.parse(req);

        const user = await this.service.assignRoleToUser(params.id, body.rolesIds);

        res.status(200).json({ message: "Role assigned to user successfully", user });
    };

    /**
     * Update a user's information by ID
     */
    public updateUser = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params, body } = this.schemas.updateUser.parse(req);

        const user = await this.service.updateUserById({
            id: params.id,
            userData: body
        });

        res.status(200).json({ message: "User updated successfully", user });
    };

    /**
     * Remove one or more roles from user
     */
    public removeRoleFromUser = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params, body } = this.schemas.removeRoleFromUser.parse(req);

        const user = await this.service.removeRoleFromUserById(params.id, body.rolesIds);

        res.status(200).json({ message: "User role removed successfully", user });
    }

    /**
     * Delete a user by ID
     */
    public deleteUser = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.deleteUser.parse(req);

        const user = await this.service.deleteUserById(params.id);

        res.status(200).json({ message: 'User deleted successfully', user });
    };

    public createStudent = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.schemas.createStudent.parse(req);

        const student = await this.service.createStudent(body);

        res.status(201).json({ message: 'Student created successfully', student });
    };

    public getStudentById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getStudentById.parse(req);

        const student = await this.service.getStudentById(params.id);

        res.status(200).json({ student });
    };

    public listStudents = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { query } = this.schemas.listAdaptedStudents.parse(req);

        const students = await this.service.listStudents(query);

        res.status(200).json({ students });
    };

    public updateStudent = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body, params } = this.schemas.updateStudent.parse(req);

        const student = await this.service.updateStudent({
            id: params.id,
            data: body,
        });

        res.status(200).json({ message: 'Student updated successfully', student });
    };

    public deleteStudent = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.deleteStudent.parse(req);

        const student = await this.service.deleteStudent(params.id);

        res.status(200).json({ message: 'Student deleted successfully', student });
    };

    public createTeacher = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body } = this.schemas.createTeacher.parse(req);

        const teacher = await this.service.createTeacher(body);

        res.status(201).json({ message: 'Teacher created successfully', teacher });
    };

    public getTeacherById = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.getTeacherById.parse(req);

        const teacher = await this.service.getTeacherById(params.id);

        res.status(200).json({ teacher });
    };

    public listTeachers = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { query } = this.schemas.getTeachers.parse(req);

        const teachers = await this.service.listTeachers(query);

        res.status(200).json({ teachers });
    };

    public updateTeacher = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { body, params } = this.schemas.updateTeacher.parse(req);

        const student = await this.service.updateTeacher({
            id: params.id,
            data: body,
        });

        res.status(200).json({ message: 'Teacher updated successfully', student });
    };

    public deleteTeacher = async (
        req: Request,
        res: Response,
    ): Promise<void> => {
        const { params } = this.schemas.deleteTeacher.parse(req);

        const student = await this.service.deleteTeacher(params.id);

        res.status(200).json({ message: 'Teacher deleted successfully', student });
    };
}