import { IUserRepository } from './interfaces/user.repository.interface';
import { IUser } from './interfaces/user.interface';
import { User } from './user.entity';
import {
    ICreateStudent,
    ICreateTeacher,
    IDeleteStudent,
    IDeleteTeacher,
    IGetStudentById,
    IGetTeacherById,
    IListRoles,
    IListStudents,
    IListTeachers,
    IParamsCreateUser,
    IParamsUpdateUser,
    IParamsUserService,
    IUpdateStudent,
    IUpdateTeacher,
    IUserService,
} from './interfaces/user.service.interface';
import { BcryptService } from '../../shared/services/bcrypt.service';
import { IRole } from '../Roles/interfaces/roles.interface';
import { IParamsCreateStudent, IParamsUpdateStudent } from '../Student/interfaces/student.service.interface';
import { IStudent } from '../Student/interfaces/student.interface';
import { IParamsCreateTeacher, IParamsUpdateTeacher } from '../Teacher/interfaces/teacher.service.interface';
import { ITeacher } from '../Teacher/interfaces/teacher.interface';

/**
 * Service responsible for business logic related to Users.
 * Handles creation, retrieval, update, and deletion of users.
 */
export class UserService implements IUserService {
    private userRepository: IUserRepository;
    private bcryptService: BcryptService;
    public listRolesDep?: IListRoles;
    public createStudentDep?: ICreateStudent;
    public getStudentByIdDep?: IGetStudentById;
    public listStudentsDep?: IListStudents;
    public updateStudentDep?: IUpdateStudent;
    public deleteStudentDep?: IDeleteStudent;
    public createTeacherDep?: ICreateTeacher;
    public getTeachersDep?: IGetTeacherById;
    public listTeachersDep?: IListTeachers;
    public updateTeacherDep?: IUpdateTeacher;
    public deleteTeacherDep?: IDeleteTeacher;

    constructor (
        { userRepository }: IParamsUserService
    ) {
        this.userRepository = userRepository;
        this.bcryptService = new BcryptService();
    }

    /**
     * Create a new user
     * @param params - The user data to create
     * @returns The created user document
     */
    public createUser = async (
        params: IParamsCreateUser
    ): Promise<IUser> => {
        const newUser = new User({
            name: params.name,
            email: params.email,
            password: params.password,
            roles: []
        });

        const existingUser = await this.userRepository.findUserByEmail(params.email);
        if (existingUser) throw new AppError('Email already in use', 400);

        const userData = newUser.getData();

        // Hash the password
        userData.password = this.bcryptService.hashPassword(params.password);

        return await this.userRepository.createUser(userData);
    }

    /**
     * Assign a role to a user
     * @param userId - The user's ID
     * @param roleIds - The role's IDs
     * @returns The user document
     */
    public assignRoleToUser = async (
        userId: string,
        roleIds: string[]
    ): Promise<IUser> => {
        if (!userId || !roleIds) {
            throw new AppError('User Id or role Id not provided', 400);
        }
        const user = await this.userRepository.findUserById(userId);
        if (!user) {
            throw new AppError('User not found', 404);
        }
        const existingRolesIds = user.roles || [];
        const rolesToAdd = roleIds.filter(roleId => !existingRolesIds.includes(roleId));
        if (rolesToAdd.length === 0) {
            throw new AppError('User already have these role', 400);
        }

        const existingRoles = await (this.listRolesDep as IListRoles)({ _id: { $in: roleIds } });
        const foundRoleIds = existingRoles.map(role => role.id);

        const newRoles = [
            ...existingRolesIds,
            ...foundRoleIds
        ] as string[];

        if (user.roles === newRoles) {
            throw new AppError('There are no roles to be added', 400);
        }

        const newUser = new User({
            ...user,
            roles: newRoles,
        })

        return await this.userRepository.updateUserById(
            userId,
            newUser.getData()
        ) as IUser;
    }

    /**
     * Get a user by ID
     * @param id - The user's ID
     * @returns The user document or null if not found
     */
    public getUserById = async (
        id: string
    ): Promise<IUser> => {
        const user = await this.userRepository.findUserById(id);
        if (!user) throw new AppError('User not found', 404);

        return user;
    }

    /**
     * Get a user by email
     * @param email - The user's email
     * @returns The user document or null if not found
     */
    public getUserByEmail = async (
        email: string
    ): Promise<IUser> => {
        const user = await this.userRepository.findUserByEmail(email);
        if (!user) throw new AppError('User not found', 404);

        return user;
    }

    /**
     * Get all roles of user
     * @param id - User's id
     * @returns Array of roles
     */
    public getRolesFromUserById = async (
        id: string,
    ): Promise<IRole[]> => {
        const user = await this.userRepository.findUserById(id);
        if (!user) throw new AppError('User not found', 404);

        if (!user.roles || user.roles.length === 0) {
            throw new AppError('User does not have roles', 400);
        }

        // Awaits the resolution of all promisses
        return await (this.listRolesDep as IListRoles)({ _id: { $in: user.roles } });
    }

    /**
     * List all users with optional filters
     * @param filter - Filters for the query
     * @returns An array of user documents
     */
    public listUsers = async (
        filter: Partial<IUser> = {}
    ): Promise<IUser[]> => {
        return await this.userRepository.listUsers(filter);
    }

    /**
     * Update a user's information by ID
     * @param params.id - The user's ID
     * @param params.userData - The data to update
     * @returns The updated user document or null if not found
     */
    public updateUserById = async (
        params: IParamsUpdateUser,
    ): Promise<IUser> => {
        const user = await this.userRepository.findUserById(params.id);
        if (!user) throw new AppError('User not found', 404);

        // Merge existing user data with new data to validate
        const newUser = new User ({
            ...user,
            ...params.userData
        });

        const userData = newUser.getData();

        // Hash the new password before updating
        if (params.userData.password) {
            userData.password = this.bcryptService.hashPassword(params.userData.password);
        }

        return await this.userRepository.updateUserById(params.id, userData) as IUser;
    }

    /**
     * Remove one or more roles of user
     * @param userId - User's ID
     * @param roleId - Role's ID
     * @returns 
     */
    public removeRoleFromUserById = async (
        userId: string,
        roleIds: string[]
    ): Promise<IUser> => {
        const user = await this.userRepository.findUserById(userId);
        if (!user) throw new AppError('User not found', 404);

        if (!user.roles || user.roles.length === 0) {
            throw new AppError('User does not have roles', 400);
        }

        const rolesToRemove = roleIds.filter(roleId => user.roles?.includes(roleId));
        if (rolesToRemove.length === 0) {
            throw new AppError('User no longer has these roles', 400);
        }

        const roles = await (this.listRolesDep as IListRoles)({ _id: { $in: roleIds }});
        if (roles.length === 0) {
            throw new AppError("Roles not found", 404);
        }

        const userRoles = user.roles?.filter(role => !rolesToRemove.includes(role));

        const newUser = new User({
            ...user,
            roles: userRoles,
        });

        return await this.userRepository.updateUserById(
            userId,
            newUser.getData(),
        ) as IUser;
    }

    /**
     * Delete a user by ID
     * @param id - The user's ID
     * @returns The deleted user document or null if not found
     */
    public deleteUserById = async (
        id: string
    ): Promise<IUser> => {
        const user = await this.userRepository.findUserById(id);
        if (!user) throw new AppError('User not found', 404);

        return await this.userRepository.deleteUserById(id) as IUser;
    }

    public createStudent = async (
        params: IParamsCreateStudent
    ): Promise<IStudent> => {
        return await (this.createStudentDep as ICreateStudent)(params);
    }
    
    public getStudentById = async (
        id: string
    ): Promise<IStudent> => {
        return await (this.getStudentByIdDep as IGetStudentById)(id);
    }
    
    public listStudents = async (
        filter: object
    ): Promise<IStudent[]> => {
        return await (this.listStudentsDep as IListStudents)(filter);
    }
    
    public updateStudent = async (
        params: IParamsUpdateStudent
    ): Promise<IStudent> => {
        return await (this.updateStudentDep as IUpdateStudent)(params);
    }
    
    public deleteStudent = async (
        id: string
    ): Promise<IStudent> => {
        return await (this.deleteStudentDep as IDeleteStudent)(id);
    }
    
    public createTeacher = async (
        params: IParamsCreateTeacher
    ): Promise<ITeacher> => {
        return await (this.createTeacherDep as ICreateTeacher)(params);
    }
    
    public getTeacherById = async (
        id: string
    ): Promise<ITeacher> => {
        return await (this.getTeachersDep as IGetTeacherById)(id);
    }
    
    public listTeachers = async (
        filter: object
    ): Promise<ITeacher[]> => {
        return await (this.listTeachersDep as IListTeachers)(filter);
    }
    
    public updateTeacher = async (
        params: IParamsUpdateTeacher
    ): Promise<ITeacher> => {
        return await (this.updateTeacherDep as IUpdateTeacher)(params);
    }
    
    public deleteTeacher = async (
        id: string
    ): Promise<ITeacher> => {
        return await (this.deleteTeacherDep as IDeleteTeacher)(id);
    }
}