import { Request, Response, NextFunction } from 'express';
import { IAuthenticatableEntity, IAuthJwtPayload } from '../../modules/auth/interfaces/auth.interface';
import { AuthService } from '../../modules/auth/auth.service';
import { IRegisterParams } from '../../modules/auth/interfaces/auth.service.interface';

/**
 * Generic authentication middleware that can work with any authenticatable entity
 * @template T - Type of entity that extends IAuthenticatableEntity
 * @template R - Type of registration parameters that extends IRegisterParams
 * @param authService - Generic authentication service
 * @returns Authentication middleware
 */
export function CreateAuthMiddleware<
    T extends IAuthenticatableEntity,
    R extends IRegisterParams
>(
    authService: AuthService<T, R>
) {
    return async (
        req: Request,
        res: Response,
        next: NextFunction
    ) => {
        // Extract the header data
        const authHeader = req.headers.authorization;
        if (!authHeader) throw new AppError('No authorization header provided', 401);

        // Extract token from Authorization bearer
        const token = authHeader.split(' ')[1];
        if (!token) throw new AppError('Invalid token', 401);

        // Verify token
        const user = await authService.verifyToken(token);
        if (!user) throw new AppError('Invalid token', 401);

        // Attach user information to request object
        req.user = (user as unknown) as IAuthJwtPayload;

        next();
    }
}