import bcrypt from "bcryptjs";

export class BcryptService {
    private saltsRounds: number;

    constructor() {
        this.saltsRounds = Number(process.env.CRYPT_SALT_ROUNDS);
        if(!this.saltsRounds) {
            throw new AppError('Environment variables not found', 500);
        }
    }

    /**
     * Hash a password
     * @param password - The password to hash
     * @returns The hashed password
     */
    public hashPassword(
        password: string
    ): string {
        if (this.isHash(password)) return password;

        return bcrypt.hashSync(
            password,
            this.saltsRounds
        );
    }

    /**
     * Verify if a password matches the secret
     * @param password - The password to verify
     * @param hash - The hashed password to compare against
     * @returns - True if password matches the secret, false otherwise
     */
    public validatePassword(
        password: string,
        hash: string,
    ): boolean {
        return bcrypt.compareSync(
            password,
            hash
        );
    }

    /**
     * Checks if the given string is already a hash.
     * @param str - The string to check.
     * @returns `true` if the string is a hash, `false` otherwise.
     */
    public isHash(
        str: string
    ): boolean {
        if (typeof str !== 'string' || str.length !== 60) {
            return false;
        }
        const regex = /^\$2[aby]\$\d{2}\$/;
        return regex.test(str);
    }
}