import { IAuthJwtPayload } from "../../modules/auth/interfaces/auth.interface";
import jwt from "jsonwebtoken";

export class JwtService {
    private readonly secretKey: jwt.Secret;
    private readonly expiresIn: string;

    constructor() {
        this.secretKey = process.env.JWT_SECRET as jwt.Secret;
        this.expiresIn = process.env.JWT_EXPIRES_IN as string;

        if (!this.secretKey || !this.expiresIn) {
            throw new AppError('Environment variables not found', 500);
        }
    }

    /**
     * Generate a JWT token
     * @param payload - The payload to include in the token
     * @returns The generated JWT token
     */
    public generateToken (
        payload: IAuthJwtPayload
    ): string {
        // Generate JWT token and return
        return jwt.sign(
            payload,
            this.secretKey,
            {
                expiresIn: this.expiresIn
            } as jwt.SignOptions
        );
    }

    /**
     * Verify a JWT token
     * @param token - The JWT token to verify
     * @returns The decoded data payload or null if verification fails
     */
    public verifyToken (
        token: string
    ): IAuthJwtPayload | undefined {
        try {
            return jwt.verify(
                token,
                this.secretKey
            ) as IAuthJwtPayload;
        } catch (err) {
            if (err instanceof jwt.TokenExpiredError) throw new AppError('Invalid token', 401);
        }
    }
}