export interface ICall {
    id: string;
    created: Date;
    updated: Date;
    deleted: Date;
    started: Date;
    ended: Date;
    expiresAt: Date;
    status: string;
    totalBillableMinutes: string;
    projectId: string;
    recording: false;
    url: string;
}

interface IDeleteCall {
    message: string;
    call: ICall;
    statusCode?: number;
}

interface IListCalls {
    pagination: object;
    data: ICall[];
}

export class StreamingService {
    private token?: string;
    private url: string;

    constructor () {
        this.token = process.env.VIDEOCHAMADA_API_TOKEN;
        this.url = `https://api.videochamada.com.br/api/calls`;

        if (!this.token) throw new AppError('Environment variables not found', 500);
    }

    /**
     * Creates a new streaming call session.
     *
     * @param {Object} data - Call creation payload.
     * @param {string} data.title - Title of the call.
     * @param {string} data.description - Description of the call.
     * @param {Date} data.expiresAt - Expiration date for the call.
     *
     * @returns {Promise<string>} - The generated call URL.
     *
     * @throws {Error} If the API request fails.
     */
    public async createCall(
        data: {
            title: string,
            description: string,
            expiresAt: Date,
        },
    ): Promise<ICall> {
        const response = await this._request({
            method: 'POST',
            endpoint: '',
            body: JSON.stringify(data),
        }) as ICall;

        return response;
    }

    /**
     * Lists all calls within a specific date range.
     *
     * @param {Object} data - Filter parameters.
     * @param {string} data.startDate - ISO date marking the beginning of the range.
     * @param {string} data.endDate - ISO date marking the end of the range.
     *
     * @returns {Promise<ICall[]>} - Array of call objects.
     *
     * @throws {Error} If the API request fails.
     */
    public async listCalls(
        data: {
            startDate: string,
            endDate: string,
        },
    ): Promise<ICall[]> {
        const calls = await this._request({
            method: 'GET',
            endpoint: `?${new URLSearchParams(data).toString()}`
        }) as IListCalls;

        return calls.data;
    }

    /**
     * Marks a call as ended and requests deletion.
     *
     * @param {string} id - The ID of the call to be ended.
     *
     * @returns {Promise<void>}
     *
     * @throws {Error} If the API request fails.
     */
    public async deleteCall(
        id: string,
    ): Promise<void> {
        const response = await this._request({
            method: 'POST',
            endpoint: `/${id}/end`
        }) as IDeleteCall;

        if (response.statusCode === 400) return;
    }

    /**
     * Performs an HTTP request to the VideoChamada API.
     *
     * @private
     * @param {Object} params - Request configuration.
     * @param {'GET' | 'POST'} params.method - HTTP method.
     * @param {string} params.endpoint - API endpoint suffix.
     * @param {BodyInit} [params.body] - Optional request body.
     *
     * @returns {Promise<ICall | IListCalls | IDeleteCall>} - Parsed JSON response.
     *
     * @throws {Error} If the HTTP request fails.
     */
    private async _request(
        params: {
            method: 'GET' | 'POST',
            endpoint: string,
            body?: BodyInit,
        },
    ): Promise<ICall | IListCalls | IDeleteCall> {
        try {
            const response = await fetch(
                `${this.url}${params.endpoint}`, {
                    headers: {
                        'Authorization': `Bearer ${this.token}`,
                        'Content-Type': 'application/json',
                    },
                    method: params.method,
                    body: params.body,
                },
            );

            return await response.json();
        } catch (err) {
            throw new AppError(`Class could not be scheduled: ${(err as Error).message}`, 500);
        }
    }
}