import { Application, Request, Response, NextFunction, ErrorRequestHandler } from 'express';
import { OpenAPIV3 } from 'express-openapi-validator/dist/framework/types';
import * as OpenApiValidator from 'express-openapi-validator';
import swaggerUi from 'swagger-ui-express';
import path from 'path';
import * as fs from 'fs';
import * as yaml from 'js-yaml';

interface ValidationError extends Error {
    status: number;
    message: string;
    errors: {
        path?: string;
        message?: string;
    }[];
}

export class SwaggerService {
    private app: Application;
    private swaggerSpec: OpenAPIV3.Document;
    private basePath: string;

    constructor (
        app: Application,
        basePath: string,
    ) {
        this.app = app;
        this.basePath = basePath;
        this.swaggerSpec = this.generateSwaggerSpec();
    }

    /**
     * Configure the Swagger validator router
     */
    public swaggerValidator() {
        if (`${process.env.CONFIGURE_OPENAPI_VALIDATOR}` === 'true') {
            this.app.use(
                OpenApiValidator.middleware({
                    apiSpec: this.swaggerSpec,
                    validateRequests: true,
                    validateResponses: false,
                }),
            );
        }
    }

    /**
     * Configure the serve Swagger UI docs
     */
    public setupSwaggerWebDocs() {
        this.app.use(`${this.basePath}/docs`,
            swaggerUi.serve,
            swaggerUi.setup(this.swaggerSpec)
        );
    }

    /**
     * Middleware for validation error
     */
    public getErrorHandler(): ErrorRequestHandler {
        return (
            err: ValidationError,
            req: Request,
            res: Response,
            next: NextFunction
        ) => {
            if (err.errors) {
                res.status(400).json({
                    message: err.message,
                    errors: err.errors,
                });
                return;
            }
            next(err);
        };
    }

    /**
     * Generate the Spec with doc files
     * @returns - Document OpenAPI configured
     */
    private generateSwaggerSpec(): OpenAPIV3.Document {
        const docsDir = path.resolve(__dirname, '../../docs/');
        const allFiles = this.findYamlFilesRecursive(docsDir);

        const componentFiles: string[] = [];
        const pathFiles: string[] = [];

        for (const file of allFiles) {
            if (file.endsWith('.component.yaml')) {
                componentFiles.push(file);
            } else if (file.endsWith('.yaml')) {
                pathFiles.push(file);
            }
        }

        const merged = {
            openapi: '3.0.2',
            info: {},
            paths: {},
            components: {
                schemas: {},
                responses: {},
                parameters: {},
                securitySchemes: {}
            }
        } as OpenAPIV3.Document;

        // Set API info from environment variables
        merged.info = {
            title: `${process.env.SWAGGER_API_TITLE}`,
            description: `${process.env.SWAGGER_API_DESCRIPTION}`,
            version: `${process.env.API_VERSION}`,
            contact: {
                name: `${process.env.SWAGGER_API_CONTACT_NAME}`,
                url: `${process.env.SWAGGER_API_CONTACT_URL}`,
                email: `${process.env.SWAGGER_API_CONTACT_EMAIL}`,
            },
            license: {
                name: `${process.env.SWAGGER_API_LICENSE_NAME}`,
                url: `${process.env.SWAGGER_API_LICENSE_URL}`,
            },
        };

        // Set the servers
        merged.servers = [
            {
                url: `${process.env.SWAGGER_API_SERVER_URL}${this.basePath}`,
                description: 'Default server'
            },
        ];

        // Merge the components
        for (const file of componentFiles) {
            const compDoc = yaml.load(fs.readFileSync(file, 'utf8')) as Partial<OpenAPIV3.Document>;

            // Deep merge in all components
            if (compDoc && compDoc.components) {
                if (compDoc.components.schemas) {
                    merged.components!.schemas = {
                        ...merged.components!.schemas,
                        ...compDoc.components.schemas
                    };
                }
                if (compDoc.components.responses) {
                    merged.components!.responses = {
                        ...merged.components!.responses,
                        ...compDoc.components.responses
                    };
                }
                if (compDoc.components.parameters) {
                    merged.components!.parameters = {
                        ...merged.components!.parameters,
                        ...compDoc.components.parameters
                    };
                }
                if (compDoc.components.securitySchemes) {
                    merged.components!.securitySchemes = { 
                        ...merged.components!.securitySchemes, 
                        ...compDoc.components.securitySchemes 
                    };
                }
            }
        }

        // Merge all paths in YAML docs
        for (const file of pathFiles) {
            const doc = yaml.load(fs.readFileSync(file, 'utf8')) as Partial<OpenAPIV3.Document>;

            if (doc && doc.paths) {
                merged.paths = {
                    ...merged.paths,
                    ...doc.paths
                };
            }
        }

        return merged;
    }

    /**
     * Recursively searches for all files ending with .yaml
     * starting from an initial directory.
     * Returns an array of absolute paths.
     */
    private findYamlFilesRecursive(dir: string): string[] {
        const allFiles: string[] = [];
        const items = fs.readdirSync(dir);

        for (const item of items) {
            const fullPath = path.join(dir, item);
            const stat = fs.statSync(fullPath);

            if (stat.isDirectory()) {

                // If it's a directory, call the function recursively
                // and add the found files to the array

                allFiles.push(...this.findYamlFilesRecursive(fullPath));

            } else if (stat.isFile() && fullPath.endsWith('.yaml')) {

                // If it's a .yaml file, add it to the array
                allFiles.push(fullPath);
            }
        }
        return allFiles;
    }
}